<p>Operating systems have global directories where any user has write access. Those folders are mostly used as temporary storage areas like
<code>/tmp</code> in Linux based systems. An application manipulating files from these folders is exposed to race conditions on filenames: a malicious
user can try to create a file with a predictable name before the application does. A successful attack can result in other files being accessed,
modified, corrupted or deleted. This risk is even higher if the application runs with elevated permissions.</p>
<p>In the past, it has led to the following vulnerabilities:</p>
<ul>
  <li> <a href="https://nvd.nist.gov/vuln/detail/CVE-2012-2451">CVE-2012-2451</a> </li>
  <li> <a href="https://nvd.nist.gov/vuln/detail/CVE-2015-1838">CVE-2015-1838</a> </li>
</ul>
<p>This rule raises an issue whenever it detects a hard-coded path to a publicly writable directory like <code>/tmp</code> (see examples bellow). It
also detects access to environment variables that point to publicly writable directories, e.g., <code>TMP</code>, <code>TMPDIR</code> and
<code>TEMP</code>.</p>
<ul>
  <li> <code>/tmp</code> </li>
  <li> <code>/var/tmp</code> </li>
  <li> <code>/usr/tmp</code> </li>
  <li> <code>/dev/shm</code> </li>
  <li> <code>/dev/mqueue</code> </li>
  <li> <code>/run/lock</code> </li>
  <li> <code>/var/run/lock</code> </li>
  <li> <code>/Library/Caches</code> </li>
  <li> <code>/Users/Shared</code> </li>
  <li> <code>/private/tmp</code> </li>
  <li> <code>/private/var/tmp</code> </li>
  <li> <code>\Windows\Temp</code> </li>
  <li> <code>\Temp</code> </li>
  <li> <code>\TMP</code> </li>
  <li> <code>%USERPROFILE%\AppData\Local\Temp</code> </li>
</ul>
<h2>Ask Yourself Whether</h2>
<ul>
  <li> Files are read from or written into a publicly writable folder </li>
  <li> The application creates files with predictable names into a publicly writable folder </li>
</ul>
<p>There is a risk if you answered yes to any of those questions.</p>
<h2>Recommended Secure Coding Practices</h2>
<p>Out of the box, .NET is missing secure-by-design APIs to create temporary files. To overcome this, one of the following options can be used:</p>
<ul>
  <li> Use a dedicated sub-folder with tightly controlled permissions </li>
  <li> Created temporary files in a publicly writable folder and make sure:
    <ul>
      <li> Generated filename is unpredictable </li>
      <li> File is readable and writable only by the creating user ID </li>
      <li> File descriptor is not inherited by child processes </li>
      <li> File is destroyed as soon as it is closed </li>
    </ul>  </li>
</ul>
<h2>Sensitive Code Example</h2>
<pre>
using var writer = new StreamWriter("/tmp/f"); // Sensitive
</pre>
<pre>
var tmp = Environment.GetEnvironmentVariable("TMP"); // Sensitive
</pre>
<h2>Compliant Solution</h2>
<pre>
var randomPath = Path.Combine(Path.GetTempPath(), Path.GetRandomFileName());

// Creates a new file with write, non inheritable permissions which is deleted on close.
using var fileStream = new FileStream(randomPath, FileMode.CreateNew, FileAccess.Write, FileShare.None, 4096, FileOptions.DeleteOnClose);
using var writer = new StreamWriter(fileStream);
</pre>
<h2>See</h2>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A01_2021-Broken_Access_Control/">Top 10 2021 Category A1 - Broken Access Control</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A5_2017-Broken_Access_Control">Top 10 2017 Category A5 - Broken Access Control</a>
  </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure">Top 10 2017 Category A3 - Sensitive Data
  Exposure</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/377">CWE-377 - Insecure Temporary File</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/379">CWE-379 - Creation of Temporary File in Directory with Incorrect Permissions</a>
  </li>
  <li> <a href="https://owasp.org/www-community/vulnerabilities/Insecure_Temporary_File">OWASP, Insecure Temporary File</a> </li>
  <li> STIG Viewer - <a href="https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222567">Application Security and
  Development: V-222567</a> - The application must not be vulnerable to race conditions. </li>
</ul>

