The C functions `calloc`, `malloc`, `realloc` and `free` should not be used in C++. 
Use the C++ operators `new` and `delete` instead.

# Why is This a Problem?

The functions `calloc`, `malloc` and `realloc` are remainder C-style functions in C++ that do not guarantee type safety, 
i.e. they return a `void*` pointer instead of pointing to an object. 
This way, you have to ensure that the `malloc` function is called with correct parameters. 
Otherwise, `malloc` may return `0` to indicate memory exhaustion.
This could result in type cast errors that may lead to critical malfunctions, because the program might expect another return value.

*Note*: You should not use `free` on objects allocated with `new` because `malloc`and `new` may allocate memory from
different heaps. 
This could result in a memory leak.

# How Can I Resolve This?

You should use the C++ operators `new` for allocation, and `delete` for de-allocation instead.
This way, the program uses the C++ constructors and destructors for memory management.
It guarantees type safety because `new` returns a pointer to a constructed object instead of a `void*` pointer.
By using `delete`, the program now calls the according destructor of the object, avoiding possible memory leaks.

*Note*: There is no equivalent for `realloc` in C++ for dynamic memory allocations.
For this case, C++ offers standard library containers such as `vector` that can grow naturally.

## Example (Before)

``` js
void bar() {
    void f* = malloc(sizeof(Foo));
    // some code
    free (f);
}
```

## Example (After)

``` js
void bar() {
    Foo* f = new Foo; 
    // some code
    delete f;
}
```

*Note*: When allocating an array using `foo *f [] = new foo[size]`, you should use `delete [] f` for de-allocation.

# Where Can I Learn More?

## Guidelines

* [C++ Core Guidelines: R.alloc: Allocation and deallocation](https://isocpp.github.io/CppCoreGuidelines/CppCoreGuidelines#ralloc-allocation-and-deallocation)

## References

* [ISO C++ FAQ: Memory Management](https://isocpp.org/wiki/faq/freestore-mgmt)

### Specific

* [ISO C++ FAQ: What is the difference between new and malloc()?](https://isocpp.org/wiki/faq/freestore-mgmt#mixing-malloc-and-delete)
* [ISO C++ FAQ: Can I free() pointers allocated with new? Can I delete pointers allocated with malloc()?](https://isocpp.org/wiki/faq/freestore-mgmt#new-vs-malloc)
* [ISO C++ FAQ: Why should I use new instead of trustworthy old malloc()?](https://isocpp.org/wiki/faq/freestore-mgmt#new-malloc-diff)