<h2>Why is this an issue?</h2>
<p>A cast is an <a
href="https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/types/casting-and-type-conversions#explicit-conversions">explicit
conversion</a>, which is a way to tell the compiler the intent to convert from one type to another.</p>
<pre>
void Method(object value)
{
    int i;
    i = (int)value;   // Casting (explicit conversion) from float to int
}
</pre>
<p>In most cases, the compiler will be able to catch invalid casts between incompatible value types or reference types.</p>
<p>However, the compiler will not be able to detect invalid casts to <a
href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/interface">interfaces</a>.</p>
<h3>What is the potential impact?</h3>
<p>Invalid casts will lead to unexpected behaviors or runtime errors such as <a
href="https://learn.microsoft.com/en-us/dotnet/api/system.invalidcastexception">InvalidCastException</a>.</p>
<h3>Exceptions</h3>
<p>No issue is reported if the interface has no implementing class in the assembly.</p>
<h2>How to fix it</h2>
<p>To prevent an <code>InvalidCastException</code> from raising during an explicit conversion, it is recommended to use the <a
href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/type-testing-and-cast#as-operator"><code>as</code> operator</a>.
When the conversion is not possible, the <code>as</code> operator returns <code>null</code> and will never raise an exception.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
public interface IMyInterface
{ /* ... */ }

public class Implementer : IMyInterface
{ /* ... */ }

public class AnotherClass
{ /* ... */ }

public static class Program
{
  public static void Main()
  {
    var another = new AnotherClass();
    var x = (IMyInterface)another;     // Noncompliant: InvalidCastException is being thrown
  }
}
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
public interface IMyInterface
{ /* ... */ }

public class Implementer : IMyInterface
{ /* ... */ }

public class AnotherClass
{ /* ... */ }

public static class Program
{
  public static void Main()
  {
    var another = new AnotherClass();
    var x = another as IMyInterface;    // Compliant: but will always be null
  }
}
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/types/casting-and-type-conversions#explicit-conversions">Casting and
  type conversions - Explicit conversion</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/type-testing-and-cast">Type-testing operators and cast
  expressions</a>
    <ul>
      <li> <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/type-testing-and-cast#is-operator"><code>is</code>
      operator</a> </li>
      <li> <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/type-testing-and-cast#as-operator"><code>as</code>
      operator</a> </li>
    </ul>  </li>
  <li> <a
  href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/language-specification/conversions#103-explicit-conversions">Conversions -
  Explicit conversions in C#</a>
    <ul>
      <li> <a
      href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/language-specification/conversions#1035-explicit-reference-conversions">Conversions - Explicit reference conversions in C#</a> </li>
    </ul>  </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/588">CWE-588 - Attempt to Access Child of a Non-structure Pointer</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/704">CWE-704 - Incorrect Type Conversion or Cast</a> </li>
</ul>

