/*
 * Decompiled with CFR 0.152.
 */
package com.teamscale.core.authenticate;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSVerifier;
import com.nimbusds.jose.crypto.ECDSAVerifier;
import com.nimbusds.jose.crypto.RSASSAVerifier;
import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.KeyType;
import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.jwt.SignedJWT;
import com.teamscale.core.user.User;
import java.text.ParseException;
import java.time.Duration;
import java.time.Instant;
import java.util.Date;
import java.util.Set;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.conqat.lib.commons.collections.Pair;
import org.conqat.lib.commons.string.StringUtils;
import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.Nullable;

public class BearerAuthenticationHelper {
    public static final String BEARER_SESSION_SUFFIX = "BEARER";
    private static final Logger LOGGER = LogManager.getLogger();
    static final String BEARER_AUTHENTICATION_PREFIX = "Bearer";
    private static final String BEARER_AUTHENTICATION_USER = "-bearer-token-";
    private static final String ENVIRONMENT_VARIABLE_NAME = "TEAMSCALE_BEARER_PUBLIC_KEY";
    public static final BearerAuthenticationHelper ADMIN_LOGIN_INSTANCE = new BearerAuthenticationHelper(System.getenv("TEAMSCALE_BEARER_PUBLIC_KEY"));
    private final boolean enabled;
    private JWK publicKey;
    private JWSVerifier verifier;

    BearerAuthenticationHelper(String publicKeyString) {
        if (publicKeyString != null) {
            try {
                this.publicKey = JWK.parse((String)publicKeyString);
                this.verifier = BearerAuthenticationHelper.buildVerifier(this.publicKey);
            }
            catch (JOSEException | ParseException e) {
                LOGGER.error("Failed to process public key passed via environment variable TEAMSCALE_BEARER_PUBLIC_KEY: " + e.getMessage(), e);
            }
        }
        this.enabled = this.publicKey != null && this.verifier != null;
    }

    private static JWSVerifier buildVerifier(JWK publicKey) throws JOSEException {
        if (KeyType.EC.equals((Object)publicKey.getKeyType())) {
            return new ECDSAVerifier(publicKey.toECKey());
        }
        if (KeyType.RSA.equals((Object)publicKey.getKeyType())) {
            return new RSASSAVerifier(publicKey.toRSAKey());
        }
        throw new JOSEException("Unsupported key type: " + String.valueOf(publicKey.getKeyType()));
    }

    public static boolean isEnabled() {
        return BearerAuthenticationHelper.ADMIN_LOGIN_INSTANCE.enabled;
    }

    public @Nullable User authenticate(@Nullable String authHeader, Pair<String, String> basicAuthenticationCredentials) {
        try {
            if (basicAuthenticationCredentials != null && BEARER_AUTHENTICATION_USER.equals(basicAuthenticationCredentials.getFirst())) {
                return this.getUserFromJwt((String)basicAuthenticationCredentials.getSecond(), EJwtTokenType.API);
            }
            if (!this.enabled || authHeader == null || !authHeader.startsWith(BEARER_AUTHENTICATION_PREFIX)) {
                return null;
            }
            return this.getUserFromJwt(StringUtils.stripPrefix((String)authHeader, (String)BEARER_AUTHENTICATION_PREFIX).trim(), EJwtTokenType.API);
        }
        catch (JOSEException | ParseException e) {
            LOGGER.debug("Failed to accept received JWT: " + e.getMessage(), e);
            return null;
        }
    }

    public @NonNull User getUserFromJwt(String token, EJwtTokenType expectedTokenType) throws ParseException, JOSEException {
        SignedJWT jwt = SignedJWT.parse((String)token);
        if (!jwt.verify(this.verifier)) {
            throw new JOSEException("Invalid signature!");
        }
        JWTClaimsSet claims = jwt.getJWTClaimsSet();
        if (claims.getAudience().size() != 1 || !claims.getAudience().contains(expectedTokenType.audience)) {
            throw new JOSEException("Invalid audience!");
        }
        String email = claims.getSubject();
        if (email == null) {
            throw new JOSEException("Subject missing!");
        }
        Date notBeforeTime = claims.getNotBeforeTime();
        if (notBeforeTime == null || Instant.now().isBefore(notBeforeTime.toInstant())) {
            throw new JOSEException("JWT not yet valid!");
        }
        Date expirationTime = claims.getExpirationTime();
        if (expirationTime == null || Instant.now().isAfter(expirationTime.toInstant())) {
            throw new JOSEException("JWT no longer valid!");
        }
        Duration tokenLifetime = Duration.between(notBeforeTime.toInstant(), expirationTime.toInstant());
        if (!tokenLifetime.minus(expectedTokenType.maxTokenLifetime).isNegative()) {
            throw new JOSEException("JWT with too long expiration!");
        }
        return BearerAuthenticationHelper.createAdminUser(email);
    }

    public @Nullable String getUserNameFromJWT(String token, String usernameClaim) throws ParseException, JOSEException {
        SignedJWT jwt = SignedJWT.parse((String)token);
        if (!jwt.verify(this.verifier)) {
            throw new JOSEException("Invalid signature!");
        }
        JWTClaimsSet claims = jwt.getJWTClaimsSet();
        Date notBeforeTime = claims.getNotBeforeTime();
        if (notBeforeTime != null && Instant.now().isBefore(notBeforeTime.toInstant())) {
            throw new JOSEException("JWT not yet valid!");
        }
        Date expirationTime = claims.getExpirationTime();
        if (expirationTime == null || Instant.now().isAfter(expirationTime.toInstant())) {
            throw new JOSEException("JWT no longer valid!");
        }
        return claims.getStringClaim(usernameClaim);
    }

    static User createAdminUser(String email) {
        User user = new User(email, "", "", email, "DenyAll");
        user.addGroupIds(Set.of("Administrators"));
        return user;
    }

    public static enum EJwtTokenType {
        API("teamscale-api", Duration.ofHours(1L)),
        LOGIN("teamscale-login", Duration.ofSeconds(10L));

        private final String audience;
        private final Duration maxTokenLifetime;

        private EJwtTokenType(String audience, Duration maxTokenLifetime) {
            this.audience = audience;
            this.maxTokenLifetime = maxTokenLifetime;
        }

        public String getAudience() {
            return this.audience;
        }

        public Duration getMaxTokenLifetime() {
            return this.maxTokenLifetime;
        }
    }
}

