<h2>Why is this an issue?</h2>
<p>Java 21 introduces the new Sequenced Collections API, which is applicable to all collections with a defined sequence on their elements, such as
<code>LinkedList</code>, <code>TreeSet</code>, and others (see <a href="https://openjdk.org/jeps/431">JEP 431</a>). For projects using Java 21 and
onwards, this API should be utilized instead of workaround implementations that were necessary before Java 21.</p>
<p>This rule reports when a collection is iterated in reverse through explicit implementation or workarounds, instead of using the reversed view of
the collection.</p>
<h2>How to fix it</h2>
<p>Replace the reported statement with a forward-iteration over the reversed view of the collection.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
void printLastToFirst(List<String> list) {
  for (var it = list.listIterator(list.size()); it.hasPrevious();) {
    var element = it.previous();
    System.out.println(element);
  }
}
```
<h4>Compliant solution</h4>

```java
void printLastToFirst(List<String> list) {
  for (var element: list.reversed()) {
    System.out.println(element);
  }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Java Documentation - <a href="https://docs.oracle.com/en/java/javase/21/docs/api/java.base/java/util/SequencedCollection.html">Interface
  SequencedCollection</a> </li>
  <li> OpenJDK - <a href="https://openjdk.org/jeps/431">JEP 431: Sequenced Collections</a> </li>
  <li> Java Documentation - <a
  href="https://docs.oracle.com/en/java/javase/21/core/creating-sequenced-collections-sets-and-maps.html#GUID-DCFE1D88-A0F5-47DE-A816-AEDA50B97523">Creating Sequenced Collections, Sets, and Maps</a> </li>
</ul>