<h2>Why is this an issue?</h2>
<p>An <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/async"><code>async</code></a> method with a
<code>void</code> return type does not follow the <a
href="https://learn.microsoft.com/en-us/dotnet/csharp/asynchronous-programming/task-asynchronous-programming-model">task asynchronous programming
(TAP)</a> model since the return type should be <a
href="https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task"><code>Task</code></a> or <a
href="https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task-1"><code>Task&lt;TResult&gt;</code></a></p>
<p>Doing so prevents control over the <a href="https://learn.microsoft.com/en-us/dotnet/csharp/asynchronous-programming/async-scenarios">asynchronous
execution</a>, such as:</p>
<ul>
  <li> waiting for the execution to complete </li>
  <li> catching any exception that might occur during execution </li>
  <li> testing execution behavior </li>
</ul>
<h3>Exceptions</h3>
<ul>
  <li> Methods with the <a href="https://learn.microsoft.com/en-us/dotnet/api/system.eventhandler"><code>EventHandler</code></a> delegate signature.  Using <code>void</code> for <code>EventHandler</code> is compliant with the TAP model.  <pre>
public async void button1_Click(object sender, EventArgs e)
{
  await DoSomethingAsync();
}
</pre> </li>
  <li> Methods name matching <code>On[A-Z]\w*</code> pattern.  Some frameworks may not use the same <code>EventHandler</code> method signature  <pre>
public async void OnClick(EventContext data)
{
  await DoSomethingAsync();
}
</pre> </li>
</ul>
<h2>How to fix it</h2>
<p>Update the return type of the method from <code>void</code> to <code>Task</code>.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
private async void ThrowExceptionAsync() // Noncompliant: async method return type is 'void'
{
  throw new InvalidOperationException();
}

public void Method()
{
  try
  {
    ThrowExceptionAsync();
  }
  catch (Exception)
  {
    // The exception is never caught here
    throw;
  }
}
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
private async Task ThrowExceptionAsync() // Compliant: async method return type is 'Task'
{
  throw new InvalidOperationException();
}

public async Task Method()
{
  try
  {
    await ThrowExceptionAsync();
  }
  catch (Exception)
  {
    // The exception is caught here
    throw;
  }
}
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/async"><code>async</code> (C#
  Reference)</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/asynchronous-programming/async-scenarios">Asynchronous
  programming</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/asynchronous-programming/task-asynchronous-programming-model">Task
  asynchronous programming model</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task"><code>Task</code> Class</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task-1"><code>Task&lt;TResult&gt;</code>
  Class</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/api/system.eventhandler"><code>EventHandler</code> Delegate</a> </li>
</ul>

