<h2>Why is this an issue?</h2>
<p>Regular expressions in JavaScript can have a global flag (<code>/g</code>) that enables global searching and matching. While this flag can be
useful in certain scenarios, it should be used with caution. When a regular expression has the global flag enabled, it remembers the position of the
last match and continues searching for subsequent matches from that position. This behavior can lead to unexpected results if you’re not careful and
be a source of bugs that are tricky to debug.</p>
<p>The global flag introduces shared state within the regular expression object. This means that if you use the same regular expression object across
multiple operations or functions, it maintains its internal state, such as the last match position.</p>

```js
const regex = /\d{4}-\d{2}-\d{2}/g;
regex.test('2020-08-06');
regex.test('2019-10-10'); // Noncompliant: the regex will return "false" despite the date being well-formed
```
<p>You should not use the global flag if you intend to use the same regular expression across multiple operations.</p>

```js
const regex = /\d{4}-\d{2}-\d{2}/;
regex.test('2020-08-06');
regex.test('2019-10-10');
```
<p>Incorrect usage of global regular expressions can result in infinite loops. For example, if you use a different instance of the same regular
expression in a <code>while</code>, it can continuously match the same substring, causing an infinite loop.</p>

```js
const input = 'foodie fooled football';
while ((result = /foo*/g.exec(input)) !== null) { // Noncompliant: a regex is defined at each iteration causing an infinite loop
  /* ... */
}
```
<p>To avoid an infinite loop, you should create the regular expression with the global flag only once, assign it to a variable, and use the same
variable in the loop.</p>

```js
const regex = /foo*/g;
const input = 'foodie fooled football';
while ((result = regex.exec(input)) !== null) {
  /* ... */
}
```
<p>Mixing the global flag (<code>g</code>) and the sticky flag (<code>y</code>) can have different effects on how regular expressions are matched and
the behavior of certain methods. The <code>test()</code> method ignores the global flag and behaves as if only the sticky flag is set.</p>

```js
const regex = /abc/gy; // Noncompliant: a regex enabling both sticky and global flags ignores the global flag
regex.test(/* ... */);
```
<p>Therefore, sou should remove the redundant global flag from the regular expression and only enable the sticky flag.</p>

```js
const regex = /abc/y;
regex.test(/* ... */);
```
<p>Overall, this rule raises an issue when:</p>
<ul>
  <li> a regular expression is tested against different inputs with <code>RegExp.prototype.test()</code> or <code>RegExp.prototype.exec()</code> </li>
  <li> a regular expression is defined within a loop condition while used with <code>RegExp.prototype.exec()</code> </li>
  <li> a regular expression turns on both global <code>g</code> and sticky <code>y</code> flags </li>
</ul>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_expressions#advanced_searching_with_flags">Regular expression flags</a>
  </li>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/RegExp/exec"><code>RegExp.prototype.exec()</code></a> </li>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/RegExp/test"><code>RegExp.prototype.test()</code></a> </li>
</ul>