Concurrent map operations that need multiple method calls can lead to race conditions.
Consider using methods specifically intended for these combined operations.

# What Does This Check Look For?

This check looks for calls to `put` or `remove` on concurrent maps after calling `containsKey`, `get` or `getOrDefault` which indicates a possible race condition.

# Why is This a Problem?

The collections in package `java.util.concurrent` are implemented for safe usage among multiple threads.
However, there are some pitfalls causing race conditions on thread interleaving between incomplete operations.
Inherited classes from `ConcurrentMap` (e.g., `ConcurrentHashMap`) enable parallel operations without corrupting the internals of the map.
However, caution is required when operations span across multiple method calls inherited from `java.util.Map`, e.g., `get()`, `put()`, `remove()`, or `containsKey()`.
A typical error is to call `put()` after `get()`.

# How Can I Resolve This?

Use combined API methods that execute atomically, e.g., `compute()`, `computeIfAbsent()`, `computeIfPresent()`, or `replace()`.

# Examples

## Non-Compliant

```java
ConcurrentHashMap<String, Integer> map = new ConcurrentHashMap<>();
// ...
Integer oldValue = map.get(key);
Integer newValue = oldValue == null ? 1 : oldValue + 1;
map.put(key, newValue);
```

The above call to `put` might not increment by one, as was intended, if another thread has modified the value associated with `key` in the meantime.

## Compliant

```java
ConcurrentHashMap<String, Integer> map = new ConcurrentHashMap<>();
// ...
map.compute(key, (unused, oldValue) -> oldValue == null ? 1 : oldValue + 1);
```

# Where Can I Learn More?

* [Code Review Checklist (Java Concurrency): Item RC.1](https://github.com/code-review-checklists/java-concurrency#chm-race)
* [Code Review Checklist (Java Concurrency): Item DC.5](https://github.com/code-review-checklists/java-concurrency#concurrent-map-type)
* [Code Review Checklist (Java Concurrency): Item DC.6](https://github.com/code-review-checklists/java-concurrency#chm-type)
