Returning in `finally` blocks can cause thrown exceptions to be lost.
The return statement should be removed.

# What Does This Check Look For?

This check detects `return` statements found in `finally` blocks.

# Why is This a Problem?
Code within `finally` blocks is executed after the execution of code in the corresponding `try` and `catch` blocks.
If a `try` or `catch` block terminates with a not yet handled exception, returning in a `finally` block can supersede handling of the in-progress exception.
The (re-)thrown exception may be discarded and not handled correctly.


Note: In the case of Objective-C, returning from a `@finally` block may prematurely free the memory of a thrown exception, leading to a zombie exception.
In C#, the occurrence of a `return` statement in a `finally` block is a compile-time error.

# How Can I Resolve This?
In general, you should avoid returning from `finally` blocks to prevent losing exceptions or superceding other return statements.


Note: In the case of Objective-C, if the `return` cannot be avoided, explicitly manage the memory for thrown exceptions, see reference below for further details.

# Examples
## Non-Compliant

```java
void nonCompliant() {
    try {
        throw new RuntimeException("Ex-1");
    } finally {
        return; // non-compliant: exception "Ex-1" will be lost
    }
}
```

## Compliant

```java
void compliant() {
    try {
        throw new RuntimeException("Ex-1");
    } finally {
       someCleanup(); // compliant: exception "Ex-1" is not lost
    }
}
```

# Where Can I Learn More?

- [Java Language Specification: Execution of try-finally and try-catch-finally](https://docs.oracle.com/javase/specs/jls/se17/html/jls-14.html#jls-14.20.2)
- [Python Enhancement Proposals: PEP 601 – Forbid return/break/continue breaking out of finally](https://peps.python.org/pep-0601/)
- [C# Specification: The throw statement](https://docs.microsoft.com/en-us/dotnet/csharp/language-reference/language-specification/statements#12106-the-throw-statement)
- [Apple Documentation: Exception Handling and Memory Management](https://developer.apple.com/library/archive/documentation/Cocoa/Conceptual/Exceptions/Tasks/HandlingExceptions.html#//apple_ref/doc/uid/20000059-SW7)
- [OWASP: Return Inside Finally Block](https://owasp.org/www-community/vulnerabilities/Return_Inside_Finally_Block)
- [CWE-584: Return Inside Finally Block](https://cwe.mitre.org/data/definitions/584.html)
