<h2>Why is this an issue?</h2>
<p>Using <code>Integer.toHexString</code> is a common mistake when converting sequences of bytes into hexadecimal string representations. The problem
is that the method trims leading zeroes, which can lead to wrong conversions. For instance a two bytes value of <code>0x4508</code> would be converted
into <code>45</code> and <code>8</code> which once concatenated would give <code>0x458</code>.</p>
<p>This is particularly damaging when converting hash-codes and could lead to a security vulnerability.</p>
<p>This rule raises an issue when <code>Integer.toHexString</code> is used in any kind of string concatenations.</p>
<h3>Noncompliant code example</h3>

```java
MessageDigest md = MessageDigest.getInstance("SHA-256");
byte[] bytes = md.digest(password.getBytes("UTF-8"));

StringBuilder sb = new StringBuilder();
for (byte b : bytes) {
    sb.append(Integer.toHexString( b & 0xFF )); // Noncompliant
}
```
<h3>Compliant solution</h3>

```java
MessageDigest md = MessageDigest.getInstance("SHA-256");
byte[] bytes = md.digest(password.getBytes("UTF-8"));

StringBuilder sb = new StringBuilder();
for (byte b : bytes) {
    sb.append(String.format("%02X", b));
}
```
<h2>Resources</h2>
<ul>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/704">CWE-704 - Incorrect Type Conversion or Cast</a> </li>
  <li> Derived from FindSecBugs rule <a href="https://find-sec-bugs.github.io/bugs.htm#BAD_HEXA_CONVERSION">BAD_HEXA_CONVERSION</a> </li>
</ul>