<h2>Why is this an issue?</h2>
<p>Logical AND (<code>&amp;&amp;</code>) operator is sometimes used to conditionally render in React (aka short-circuit evaluation). For example,
<code>myCondition &amp;&amp; &lt;MyElement /&gt;</code> will return <code>&lt;MyElement /&gt;</code> if <code>myCondition</code> is <code>true</code>
and <code>false</code> otherwise.</p>
<p>React considers <code>false</code> as a 'hole' in the JSX tree, just like <code>null</code> or <code>undefined</code>, and doesn’t render anything
in its place. But if the condition has a <code>falsy</code> non-boolean value (e.g. <code>0</code>), that value will leak into the rendered
result.</p>
<p>This rule will report when the condition has type <code>number</code> or <code>bigint</code>.</p>
<p>In the case of React Native, the type <code>string</code> will also raise an error, as your render method will crash if you render <code>0</code>,
<code>''</code>, or <code>NaN</code>.</p>

```ts
function Profile(props) {
  return <div>
    <h1>{ props.username }</h1>
    { props.orders && <Orders /> } { /* Noncompliant: 0 will be rendered if no orders available */ }
  </div>;
}
```
<p>Instead, make the left-hand side a boolean to avoid accidental renderings.</p>

```ts
function Profile(props) {
  return <div>
    <h1>{ props.username }</h1>
    { props.orders > 0 && <Orders /> }
  </div>;
}
```
<p>Another alternative to achieve conditional rendering is using the ternary operator (<code>myCondition ? &lt;MyElement /&gt; : null</code>), which
is less error-prone in this case as both return values are explicit.</p>

```ts
function Profile(props) {
  return <div>
    <h1>{ props.username }</h1>
    { props.orders ? <Orders /> : null }
  </div>;
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> React Documentation - <a href="https://react.dev/learn/conditional-rendering#logical-and-operator-">Conditional Rendering</a> </li>
</ul>