# Rule 23.6 The controlling expression of a generic selection shall have an essential type that matches its standard type

## Category
Required

## Analysis
Decidable, Single Translation Unit

## Applies to
C11

## Amplification
Using an expression that has a distinct essential type from its standard type as the controlling
expression of a generic selection is always a violation of this rule, except for integer constant
expressions that are neither character constants nor essentially boolean.

*Notes*:
1. An enumeration constant unconditionally has the type `signed int` and will not match
either the implementation defined underlying type, if different from `int`, or the
(indistinguishable) enumerated type, whichever is listed.
2. The same consideration applies to `true` and `false` defined in `<stdbool.h>`, which have
type `int` and will not match an association of type `_Bool`.

## Rationale
The association selected by a generic selection must be in line with the type system under which the
code was designed, in order to be useful and consistent. Code written under MISRA guidelines is
subject to the essential type system, and therefore if the selected type is not consistent with the
essential type of the controlling expression, the generic selection will have violated the type system
used for the design.

Because generic selections choose a standard type by name, they can expose this difference if used
with an argument that has inconsistent essential and standard types.

## Exception
This rule does not apply to integer constant expressions which would have an essentially signed or
unsigned type of lower rank than `int`, because of the *Type of Lowest Rank* rule, but are neither
character constants nor essentially boolean.

## Example
The non-compliant examples below accept an operand with essential type of `signed short` and
`char` respectively, but both have a standard type of `signed int`, and only the standard type
determines which association will match.

An exception is made for integer constant expressions that are not character constants, because this
is in line with most user expectations.
```c
#define filter_ints(X) ( _Generic((X) \
 , signed short: handle_sshort \
 , unsigned short: handle_ushort \
 , signed int : handle_sint \
 , unsigned int : handle_uint \
 , signed long : handle_slong \
 , unsigned long : handle_ulong \
 , default : handle_any) (X) )

short s16 = 0;
int i32 = 0;
long l32 = 0;

/* Non-compliant */
filter_ints (s16 + s16); /* Selects int, essentially short */
filter_ints ('c'); /* Selects int, essentially char */

/* Compliant */
filter_ints (s16);
filter_ints (i32);
filter_ints (l32);

/* Compliant by exception */
filter_ints (10u); /* UTLR is unsigned char */
filter_ints (250 + 350); /* STLR is signed short */
```

Because an enumerated type is compatible with its implementation-defined underlying type, it is not
distinguishable from the underlying type by `_Generic` (although distinguishing different enumerated
types is possible because this compatibility is not transitive).

Attempting to list both would be a constraint violation. Therefore, using an enumerated type as the
controlling expression to a generic selection that lists the underlying type, or vice versa, is always a
violation.
```c
enum E { A = 0, B = 1, C = 2 };
enum E e = A;

/* Non-compliant, assuming compiler chooses 'unsigned int' as the underlying type */
filter_ints (e); /* selects 'unsigned int', essentially enum */
filter_ints (A); /* selects 'signed int', essentially enum */
```

## See also
Rule 23.5

---

Copyright The MISRA Consortium Limited © [Date - April 2023].
