# Rule 20.7 Expressions resulting from the expansion of macro parameters shall be appropriately delimited

## Category
Required

## Analysis
Decidable, Single Translation Unit

## Applies to
C90, C99, C11

## Amplification
If the expansion of any macro parameter produces a token, or sequence of tokens, that form an expression then that expression, in the fully-expanded macro, shall either:
*   Be a parenthesized expression itself; or
*   Be delimited by parentheses, square brackets, braces, or commas.

*Note:* this does not necessarily require that all macro parameters are delimited; it is acceptable for delimiters to be provided in macro arguments.

## Rationale
If appropriate delimiters are not used, then operator precedence may not give the desired results when macro substitution occurs.

If a macro parameter is not being used as an expression then the delimiters are not necessary because no operators are involved.

## Example
In the following non-compliant example:
```c
#define M1( x, y ) ( x * y )

r = M1 ( 1 + 2, 3 + 4 );
```
the macro expands to give:
`r = ( 1 + 2 * 3 + 4 );`
The expressions `1 + 2` and `3 + 4` are derived from expansion of parameters `x` and `y` respectively, but neither is delimited, for example with parentheses.. The value of the resulting expression is 11, whereas the result 21 might have been expected.

The code could be written in a compliant manner either by parenthesizing the macro arguments, or by writing an alternative version of the macro that inserts parentheses during expansion, for example:
```c
r = M1 ( ( 1 + 2 ), ( 3 + 4 ) );   /* Compliant */

#define M2( x, y ) ( ( x ) * ( y ) )

r = M2 ( 1 + 2, 3 + 4 );           /* Compliant */
```
The following example is compliant because the first expansion of `x` is as the operand of the `##` operator, which does not produce an expression. The second expansion of `x` is as an expression which is delimited with parentheses as required.
```c
#define M3( x ) a ## x = ( x )

int16_t M3 ( 0 );
```
The following example is compliant because expansion of the parameter `M` as a member name does not produce an expression. Expansion of the parameter `S` produces an expression, with structure or union type, which does require delimiting.
```c
#define GET_MEMBER( S, M ) ( S ).M

v = GET_MEMBER ( s1, minval );
```
The following compliant example shows that it is not always necessary to delimit every instance of a parameter, although this is often the easiest method of complying with this rule.
```c
#define F( X ) G( X )
#define G( Y ) ( ( Y ) + 1 )

int16_t x = F ( 2 );
```
The fully-expanded macro is `( ( 2 ) + 1 )`. Tracing back through macro expansion, the value `2` arises from expansion of parameter `Y` in macro `G` which in turn arises from parameter `X` in macro `F`. Since `2` is parenthesized in the fully-expanded macro, the code is compliant.

The following compliant example shows that delimiters other than parentheses may be used. All occurences of `array` and `index` in the expansion of a `CALL` macro are properly delimited.
```c
#define CALL(A, I) f( I, A[ I ] )

r = CALL ( array, index )
```

## See also
Dir 4.9

---

Copyright The MISRA Consortium Limited © [Date - March 2025].