# Rule 5.10 A reserved identifier or reserved macro name shall not be declared

## Category
Required

## Analysis
Decidable, Single Translation Unit

## Applies to
C90, C99, C11

## Amplification
This rule applies to the following:
*   Identifiers or macro names beginning with an underscore;
*   Identifiers in file scope described in Section 7, “Library”, of the C Standard;
*   Macro names described in Section 7, “Library”, of the C Standard as being defined in a standard header.

This rule does **not** include those identifiers or macro names that are described in the section of the applicable C Standard entitled “Future Library Directions”.

## Rationale
The implementation is permitted to rely on reserved identifiers behaving as described in the C Standard and may treat them specially. If reserved identifiers are reused, the program may exhibit undefined behaviour.

## Example
In the following non-compliant example, the function *memcpy* is declared explicitly. The compliant method of declaring this function is to include `<string.h>`:

```c
/*
 * Include <stddef.h> to define size_t
 */
#include <stddef.h>

extern void *memcpy ( void *restrict s1, const void *restrict s2, size_t n );
```

An implementation is permitted to provide a *function-like macro* definition for each Standard Library function in addition to the library function itself. This feature is often used by compiler writers to generate efficient inline operations in place of the call to a library function. Using a *function-like macro*, the call to a library function can be replaced with a call to a reserved function that is detected by the compiler’s code generation phase and replaced with the inline operation. For example, the fragment of `<math.h>` that declares *sqrt* might be written using a *function-like macro* that generates a call to `_BUILTIN_sqrt` which is replaced with an inline `SQRT` instruction on processors that support it:

```c
extern double sqrt ( double x );

#define sqrt( x ) ( _BUILTIN_sqrt ( x ) )
```

The following non-compliant code might interfere with the compiler’s built-in mechanism for handling *sqrt* and therefore produce undefined behaviour:

```c
static double _BUILTIN_sqrt ( double x )    /* Non-compliant */
{ 
  return x * x;
}

#include <math.h>

double d = sqrt ( ( double ) 2.0 );         /* sqrt may not behave as
                                             * defined in the C Standard */
```

## See also
Rule 20.15

---

Copyright The MISRA Consortium Limited © [Date - March 2025].