# Rule 0.1.1 A value should not be *unnecessarily written* to a local object

## Category
Advisory

## Analysis
Undecidable, System

## Amplification
This rule applies to all accesses, either direct or through a pointer or reference, to objects with
*automatic storage duration* that:
1.  Have *trivially destructible* types (including basic types and enumeration types); or
2.  Are arrays of *trivially destructible* types; or
3.  Are STL containers (including `std::string`), where the *value_type* is *trivially destructible*.

The rule also applies to accesses to subobjects or elements of such objects.

An object is *unnecessarily written* when on each feasible path:
1.  The object is destroyed before being *observed*; or
2.  The object is written to again before being read.

An object is *observed* within an expression if its value affects the external state of the program, the
control flow of the program, or the value of a different object.

The following examples illustrate different types of access to an object `i`:
```cpp
int32_t f( int32_t j );

int32_t i = f( 1 ); // Written
i; // Read
i = 0; // Written (even if 'i' was 0 before the assignment)
auto j = i; // Read and observed
++i; // Read and written
i += 3; // Read and written
i = i + j; // Read and written
auto k1 = ++i; // Read, written, read and observed
auto k2 = i++; // Read, observed and written
arr[ i ] = f( 1 ); // Read and observed
if ( i ) { } // Read and observed
( void )f( i ); // Read and observed
```

*Observing* any element of a container is considered to *observe* the full container and all of its elements.
*Observing* a subobject is considered to *observe* the full object and all of its subobjects. Additionally, an
object that is created outside of an *iteration statement* is considered to be *observed* (but not read) at
the end of the *iteration statement*, provided it is also *observed* during any iteration.

A function’s compliance with this rule is determined independently of the context in which the function
is called. For example, a Boolean parameter is treated as if it may have a value of `true` or `false`,
even if all the calls expressed in the current program use a value of `true` — see example `f4`, below.

## Rationale
Giving an object a value that is never subsequently used is inefficient, and may indicate that a coding
defect is present. Such writes are referred to as *dataflow anomalies*:
1.  A *DU (Define–Use) dataflow anomaly* is present if a value that is written is *never observed*;
2.  A *DD (Define–Define) dataflow anomaly* is present if a value overwrites another value before it
    has been read.

Within a loop, a value may be written to an object with the intent that it will be *observed* during the
next iteration, meaning that the value written on the last iteration may never be *observed*. Whilst it is
possible to restructure the loop to avoid this behaviour, there is a risk that the resulting code may be
of lower quality (less clear, for example). This rule therefore considers *observation* during any iteration
to apply to all values written to such an object, including a value written during the last iteration of a
loop that is not actually *observed* — see example `f3`, below.

*Observing* part of a bigger object is considered to *observe* the object in its entirety; it is common to have
code that operates on objects as a whole (initializing or writing to all subobjects), even if the value of
only some of its subobjects are actually read. Requiring fine-grained writes would break encapsulation
— see examples `f5` and `f6`, below.

A function, assuming its preconditions are respected, should always behave as specified. This is true
irrespective of the calling context, including possible contexts that are not expressed in the current
program. For this reason, path feasibility (within this rule) is determined without taking the actual calling
contexts into consideration.

## Exception
Even though the values passed as arguments to functions are written to their corresponding parameter
objects, it is permitted for function parameters to remain *unobserved* when the function returns.
This exception prevents crosstalk with Rule 0.2.2 which requires, in a decidable way, that function
parameters are used. Note that writing to an unread parameter in a function body is a DD anomaly,
which is a violation of this rule.

## Example
```cpp
int32_t f1( int32_t i )
{
  auto j = i; // Non-compliant - j is not observed after being written

  i++; // Non-compliant - i is not observed after being written

  return 0;
}

int32_t f2( int32_t i )
{
  auto & j = i; // Rule does not apply to j, which is not an object

  j++; // Compliant - writes object i that is observed in the return

  return i;
}

int32_t f3( int32_t j, int32_t k, int32_t m )
{
  for ( int32_t i = 0; i < 10; ++i ) // Compliant - i is observed in i < 10
  {
    m = 10; // Non-compliant - when looping, overwrites incremented value
    ++k; // Non-compliant - k is never observed

    use( j ); // Observation of j inside of the loop

    ++j; // Compliant - observation above is sufficient for final write

    ++m; // Compliant - observed in the return
  } // j is considered observed here as it was observed in the loop

  return m;
}

int32_t f4( bool b,
            int32_t i,
            int32_t j ) // Compliant by exception - j is never observed
{
  i = 0; // Non-compliant - value passed is overwritten

  int32_t k = 4; // Compliant - value is observed in one feasible path

  if ( b ) // Both branches are considered feasible, even if the function
  { // is only called with b set to true
    return i;
  }
  else
  {
    return k;
  }
}

struct Point { int32_t x; int32_t y; int32_t z; int32_t t; };

int32_t f5()
{
  Point p {}; // Compliant - p and its subobjects are observed in the return

  p.x = 2;
  p.x = 3; // Non-compliant - overwrite the value 2 that is never read
  p.z = 4; // Compliant - p.z is observed in the return

  return p.y; // Observation of p.y also observes p, p.x, p.z and p.t
}

int32_t f6()
{
  std::vector< int32_t > v( 4, 0 ); // Compliant - v and its elements are observed
                                    // in the return

  v[ 0 ] = 2;
  v[ 0 ] = 3; // Non-compliant - overwrite the value 2 that is never read
  v[ 2 ] = 4; // Compliant - v[ 2 ] is observed in the return

  return v[ 1 ]; // Observation of v[ 1 ] observes v and all of its elements
}

void f7( std::mutex & m )
{
  std::scoped_lock lock { m }; // Rule does not apply - destructor is non-trivial
}

char f8( bool b )
{
  char c = f( 1 ); // Non-compliant - assigned value never read

  if ( b )
  {
    c = 'h'; // The value of c is overwritten here

    return c;
  }
  else
  {
    return '\0'; // The value of c is not observed here
  }
}

void callee( int32_t & ri )
{
  ri++; // Rule does not apply - reference is not an object
}

void caller()
{
  int32_t i = 0;

  callee( i ); // Non-compliant - i written and not subsequently observed
}
```

---

Copyright The MISRA Consortium Limited © [Date - October 2023].
