# Rule 0.2.4 Functions with *limited visibility* should be *used* at least once

## Category
Advisory

## Analysis
Decidable, System

## Amplification
A function has *limited visibility* if it:
1.  Is declared in an anonymous namespace; or
2.  Is a member of a class in an anonymous namespace; or
3.  Has namespace scope and is declared `static`; or
4.  Is a `private`, non-`virtual` member.

A function is *used* when:
1.  Its address is taken (including by reference); or
2.  It is called; or
3.  It is an operand of an expression in an unevaluated context; or
4.  Another function in the same overload set is *used*.

This rule does not apply to:
1.  *Special member functions*;
2.  Functions defined as `= delete`.

## Rationale
Functions with *limited visibility* are not generally used within an extensible API. If they are present but
remain *unused*, then there may be an issue in the software design.

Unused functions in an overload set are acceptable as it allows the set to be internally consistent.

## Exception
Functions that have at least one *declaration* with the `[[maybe_unused]]` attribute are permitted to
be *unused* as the intent is explicit.

## Example
```cpp
struct Foo
{
  int32_t m1() // Public - rule does not apply
  {
    return -1;
  }

  static int32_t m2() // Class scope - rule does not apply
  {
    return 42;
  }

  Foo()
  {
    m3();
  }

private:
  void m3() { } // Compliant - called
  void m4() { } // Non-compliant - not used
  void m5() { } // Compliant - used by a friend

  friend void ( *f4() )();

protected:
  void m6() { } // Protected - rule does not apply
};

static void f1() { } // Non-compliant - not used

namespace
{
  void f2() { } // Non-compliant - not used
}

static void f3() { } // Compliant - address taken in f4()

void ( *f4() )() // Rule does not apply - visibility not limited
{
  Foo bar;
  bar.m5();

  return &f3;
}

namespace A
{
  struct C1 {};
  static void swap( C1 &, C1 & ); // Compliant - overload set for call in f5
}

namespace B
{
  struct C2 {};
  static void swap( C2 &, C2 & ); // Non-compliant
}

namespace
{
  template< typename T >
  void swap( T &, T & ); // Compliant - overload set for call in f5
}

void f5( A::C1 c1, A::C1 c2 ) // Rule does not apply - visibility not limited
{
  swap( c1, c2 );
}
```

---

Copyright The MISRA Consortium Limited © [Date - October 2023].
