# Rule 19.3.4 Parentheses shall be used to ensure macro arguments are expanded appropriately

## Category
Required

## Analysis
Decidable, Single Translation Unit

## Amplification
For the purposes of this rule, a *critical operator* is an operator that has a ranking between 2 and 13
(inclusive), as specified in the table to Rule 8.0.1.

A macro argument containing a *top-level* token (see definition below) that expands as a *critical operator*
is inappropriately expanded if, within the macro definition, there is an occurrence of the corresponding
macro parameter that is not:
*   Directly parenthesized (a parameter `x` is directly parenthesized in `( x )`); or
*   Stringified (used as an operand to the `#` operator).

When a macro is expanded, a *level* can be associated with every token in the expansion of a macro
argument. For each argument, the *level* of its first token is zero, and then the *level* of each of its
subsequent tokens relative to the level of the previous token is:
*   One more, if the previous token is `(`
*   One less, if the previous token is `)`
*   The same, for any other previous token.

A token is said to be *top-level* when its *level* is less than or equal to zero.

## Rationale
When a macro is invoked with an argument that looks like an expression, it is generally assumed that
this expression will behave as if it were an argument to a function call — in particular, that it will be
evaluated in isolation.

However, since macro expansion result in textual replacement, a macro parameter is simply replaced
by the text corresponding to the argument. This means that the different tokens that form the
argument can end up forming parts of different sub-expressions. This typically happens when the
argument contains an operator having a low precedence, and the parameter is expanded next to an
operator having a higher precedence. This behaviour can generally be avoided by adding parentheses
around the macro parameter.

## Example
In the following example, the operator `+` is a *top-level* token in the `x` argument to the macro. However, `x`
is neither parenthesized nor stringified in the macro definition. The value of the resulting expression
is 7, whereas the value 9 might have been expected.

```cpp
#define M1( x, y ) ( x * y )

r = M1( 1 + 2, 3 ); // Non-compliant - x not parenthesized
 // Expands as r = ( 1 + 2 * 3 );
```

Ideally, the above can be re-written in a compliant manner by parenthesizing the macro parameters
in the macro definition:

```cpp
#define M2( x, y ) ( ( x ) * ( y ) )

r = M2( 1 + 2, 3 ); // Compliant - x is directly parenthesized
 // Expands as r = ( ( 1 + 2 ) * ( 3 ) );
```

If this is not possible, it is also acceptable to parenthesize the macro argument:

```cpp
r = M1( ( 1 + 2 ), 3 ); // Compliant - operator + is not top-level
 // Expands as r = ( ( 1 + 2 ) * 3 );
```

In the following example, the macro `M1` is invoked with `1 + 2` as its `x` parameter, and the top level `+`
token is a *critical operator*. Therefore, `x` is inappropriately expanded, as it is neither parenthesized
nor stringified in the macro definition.

```cpp
#define M3( z ) z + 2

r = M1( M3( 1 ), 3 ); // Non-compliant - operator + is top-level
 // Expands as r = ( 1 + 2 * 3 );
```

---

Copyright The MISRA Consortium Limited © [Date - October 2023].
