# Rule 6.9.2 The names of the *standard signed integer types* and *standard unsigned integer types* should not be used

## Category
Advisory

## Analysis
Decidable, Single Translation Unit

## Amplification
This rule applies to the names of integral types constructed using the keywords `char`, `short`, `int`,
`long`, `signed` and `unsigned` (ignoring any *cv-qualification*). It does not apply to the use of plain
`char`.

## Rationale
It is *implementation-defined* how much storage is required for any object of the *standard signed integer
types* or *standard unsigned integer types*. When the amount of storage being used is important, the
use of types having specified widths makes it clear how much storage is being reserved for each object.

The C++ Standard Library *header file* `<cstdint>` often provides a suitable set of integer types having
specified widths. If a project defines its own type aliases, it is good practice to use `static_assert`
to validate any size assumptions. For example:
```cpp
using torque_t = unsigned short;
static_assert( sizeof( torque_t ) >= 2 );
```

Notes:
1.  Compliance with this rule does not prevent integer promotion, which is influenced by the
    implemented size of `int` and the type used for an alias. For example, an expression of type
    `int16_t` will only be promoted if the aliased type has a rank lower than that of `int`. The
    presence or absence of integer promotion may have an influence on overload resolution.
2.  Strong typing, which may be provided by `class` or `enum` types, is more robust than the use of
    type aliases to represent specific width types.

## Exception
1.  The names of the *standard signed integer types* and *standard unsigned integer types* may be used
    to define a type alias.
2.  The name `int` may be used for:
    a.  The parameter to a postfix operator, which **must** use that type; and
    b.  The return type of `main`; and
    c.  The `argc` parameter to `main`.

## Example
```cpp
#include <cstdint>

int x = 0; // Non-compliant - use of int
int32_t y = 0; // Compliant
int_least32_t z = 0; // Compliant

using torque_t = int; // Compliant by exception #1
torque_t w = 0;

class C
{
public:
  C operator++( int ); // Compliant by exception #2.1
};

int main() { } // Compliant by exception #2.2
int main( int argc, char * argv[] ) { } // Compliant by exception #2.2 and #2.3
```

---

Copyright The MISRA Consortium Limited © [Date - October 2023].
