# Rule 12.1 The precedence of operators within expressions should be made explicit

## Category
Advisory

## Analysis
Decidable, Single Translation Unit

## Applies to
C90, C99

## Amplification
The following table is used in the definition of this rule.

| Description | Operator or Operand | Precedence |
| :--- | :--- | :--- |
| Primary | identifier, constant, string literal, ( expression ) | 16 (high) |
| Postfix | [ ] ( ) (function call) . -> ++ (post-increment) -- (post-decrement) ( ) { } (C99: compound literal) | 15 |
| Unary | ++ (pre-increment) -- (pre-decrement) & * + - ~ ! sizeof defined (preprocessor) | 14 |
| Cast | ( ) | 13 |
| Multiplicative | * / % | 12 |
| Additive | + - | 11 |
| Bitwise shift | << >> | 10 |
| Relational | < > <= >= | 9 |
| Equality | == != | 8 |
| Bitwise AND | & | 7 |
| Bitwise XOR | ^ | 6 |
| Bitwise OR | | | 5 |
| Logical AND | && | 4 |
| Logical OR | || | 3 |
| Conditional | ?: | 2 |
| Assignment | = *= /= %= += -= <<= >>= &= ^= |= | 1 |
| Comma | , | 0 (low) |

The precedences used in this table are chosen to allow a concise description of the rule. They are not necessarily the same as those that might be encountered in other descriptions of operator precedence.

For the purposes of this rule, the precedence of an expression is the precedence of the element (operand or operator) at the root of the parse tree for that expression.

For example: the parse tree for the expression a << b + c can be represented as:

```c
   <<
  /  \
 a    +
     / \
    b   c
```

The element at the root of this parse tree is '<<' so the expression has precedence 10.

The following advice is given:
*   The operand of the sizeof operator should be enclosed in parentheses;
*   An expression whose precedence is in the range 2 to 12 should have parentheses around any operand that has both:
    – Precedence of less than 13, and
    – Precedence greater than the precedence of the expression.

## Rationale
The C language has a relatively large number of operators and their relative precedences are not intuitive. This can lead less experienced programmers to make mistakes. Using parentheses to make operator precedence explicit removes the possibility that the programmer’s expectations are incorrect. It also makes the original programmer’s intention clear to reviewers or maintainers of the code.

It is recognized that overuse of parentheses can clutter the code and reduce its readability. This rule aims to achieve a compromise between code that is hard to understand because it contains either too many or too few parentheses.

## Examples
The following example shows expressions with a unary or postfix operator whose operands are either primary-expressions or expressions whose top-level operators have precedence 15.

```c
a[ i ]->n;      /* Compliant - no need to write ( a[ i ] )->n        */
*p++;           /* Compliant - no need to write *( p++ )             */
sizeof x + y;   /* Non-compliant - write either sizeof ( x ) + y
                 * or sizeof ( x + y )                               */
```

The following example shows expressions containing operators at the same precedence level. All of these are compliant but, depending on the types of a, b and c, any expression with more than one operator may violate other rules.

```c
a + b;
a + b + c;
( a + b ) + c;
a + ( b + c );
a + b - c + d;
( a + b ) - ( c + d );
```

The following example shows a variety of mixed-operator expressions:

```c
/* Compliant - no need to write f ( ( a + b ), c ) */
x = f ( a + b, c );

/* Non-compliant
 * Operands of conditional operator (precedence 2) are:
 *   ==  precedence 8  needs parentheses
 *   a   precedence 16 does not need parentheses
 *   -   precedence 11 needs parentheses
 */
x = a == b ? a : a - b;

/* Compliant */
x = ( a == b ) ? a : ( a - b );

/* Compliant
 * Operands of << operator (precedence 10) are:
 *   a   precedence 16 does not need parentheses
 *   ( E ) precedence 16 already parenthesized
 */
x = a << ( b + c );

/* Compliant
 * Operands of && operator (precedence 4) are:
 *   a   precedence 16 does not need parentheses
 *   &&  precedence 4  does not need parentheses
 */
if ( a && b && c )
{
}

/* Compliant
 * Operands of && operator (precedence 4) are:
 *   defined(X) precedence 14 does not need parentheses
 *   (E)        precedence 16 already parenthesized
 */
#if defined ( X ) && ( ( X + Y ) > Z )

/* Compliant
 * Operands of && operator (precedence 4) are:
 *   !defined ( X )  precedence 14 does not need parentheses
 *   defined ( Y )   precedence 14 does not need parentheses
 * Operand of ! operator (precedence 14) is:
 *   defined ( X )   precedence 14 does not need parentheses
 */
#if !defined ( X ) && defined ( Y )
```

Note: this rule does not require the operands of a , operator to be parenthesized. Use of the , operator is prohibited by Rule 12.3.

```c
x = a, b; /* Compliant - parsed as ( x = a ), b */
```

---

Copyright The MISRA Consortium Limited © [Date - March 2013].
