# Rule 22.9 The value of errno shall be tested against zero after calling an errno-setting-function

## Category
Required

## Analysis
Undecidable, System

## Applies to
C90, C99, C11

## Amplification
An *errno-setting-function* is one of those described in Rule 22.8.

The test of `errno` shall occur in the same function on all paths from the call of interest, and before
any subsequent function calls.

The results of an *errno-setting-function* shall not be used prior to the testing of `errno`.

## Rationale
An *errno-setting-function* writes a non-zero value to `errno` if an error is detected, leaving the value
unmodified otherwise. The C Standard includes non-normative advice that “a program that uses
errno for error checking should set it to zero before a library function call, then inspect it before a
subsequent library function call”.

As the value returned by an *errno-setting-function* is unlikely to be correct when `errno` is non-zero,
the program shall test `errno` to ensure that it is appropriate to use the returned value.

## Exception
The value of `errno` does not have to be tested when the return value of an *errno-setting-function* can
be used to determine if an error has occurred.

## Example
```c
void f1 ( void )
{
 errno = 0;
 errnoSettingFunction1 ();
 someFunction (); /* Non-compliant - function call */

 if ( 0 != errno )
 {
 }

 errno = 0;
 errnoSettingFunction2 ();

 if ( 0 != errno ) /* Compliant */
 {
 }
}

void f2 ( FILE *f, fpos_t *pos )
{
 errno = 0;
 if ( fsetpos ( f, pos ) == 0 )
 {
 /* Compliant by exception - no need to test errno as no out-of-band error
 reported. */
 }
 else
 {
 /* Something went wrong - errno holds an implementation-defined positive value.
 */
 handleError ( errno );
 }
}
```

## See also
Rule 22.8, Rule 22.10

---

Copyright The MISRA Consortium Limited © [Date - April 2023].
