# Rule 23.4 A generic association shall list an appropriate type

## Category
Required

## Analysis
Decidable, Single Translation Unit

## Applies to
C11

## Amplification
The controlling expression of a generic selection undergoes *lvalue conversion* before having its type
compared to the entries in the association list. The association list shall not contain any associations
for:
*   A const-qualified object type
*   A volatile-qualified object type
*   An atomic object type
*   An array type
*   A function type
*   An unnamed structure or union definition

## Rationale
Since the C Standard does not impose a constraint limiting the types in the association list to the
possible types of values after conversion, it is possible to list associations for types that would never
be eligible for selection.

Value conversion removes top-level qualification and “decays” array and function values into pointers.
Therefore, such types can never match the type of the converted value of the controlling expression.
Listing them is not a constraint violation, but serves no useful purpose, and is almost certainly an
error. This only affects object types, not qualification on pointed-to types.

Listing an unnamed structure or union in the association list is a violation of this rule because every
occurrence of a *struct-declaration-list* creates a distinct type, therefore it can only be matched by a
default association.

## Example
In this set of examples, the first pair are non-compliant because they explicitly specify generic
associations for types that can never be the type of the controlling expression, while the second pair
are compliant as they only attempt to specify generic associations for types that can match the
controlling expression:
```c
typedef int32_t Func (int);
typedef int32_t Array [10];

typedef Func * FuncP;
typedef Array * ArrayP;

/* Non-compliant (because Func is listed) */
#define handle_function_nc(X) _Generic((X) \
 , Func : handle_funcp (&(X)) \
 , FuncP: handle_funcp (X) )

/* Non-compliant (because Array is listed) */
#define handle_array_nc(X) _Generic((X) \
 , Array : handle_intp ((X) + 0) \
 , ArrayP : handle_intp (*(X)) \
 , int32_t *: handle_intp (X) )

/* Compliant */
#define handle_function(X) _Generic( (X), FuncP: handle_funcp (X) )

/* Compliant */
#define handle_array(X) _Generic(&(X)[0] \
 , int32_t *: handle_intp (X) \
 , ArrayP : handle_intp (*(X)) )

Array arr1;
Array arr2[10]; /* Two-dimensional - array of arrays */

handle_array (arr1); /* type of &(arr1[0]) is int32_t * - can pass to handle_intp */
handle_array (arr2); /* type of &(arr2[0]) is ArrayP - must dereference again */
```

In this set of examples, the first example is non-compliant because it explicitly specifies a type that
includes qualifiers at the object level; qualification is removed by value conversion, so such types are
not selectable. The second example is able to take qualification into account because the qualifiers
apply to the pointed-to type, not the object type, and are not removed from the type of the value of
the controlling expression:
```c
typedef int32_t Int;
typedef int32_t const CInt;

/* Non-compliant */
#define filter_const_nc(X) ( _Generic((X) \
 , CInt : handle_const_intp \
 , default: handle_other_value ) (&(X)) )

/* Compliant */
#define filter_const(X) ( _Generic((X) \
 , CInt * : handle_const_intp \
 , Int * : handle_const_intp \
 , default: handle_other_value ) (X) )
```

In this set of examples, the first example is non-compliant because it tries to explicitly specify array
types, which are not selectable, in an attempt to involve dependent type information in the match.
The second example below is compliant because it explicitly specifies a pointer type, and forces the
type of the controlling expression to be a pointer with `&`. Creating a pointer to an array preserves the
complete array type, including its dimension.
```c
/* Avoid multiple associations if size == 1 */
#define SizeofNonEmpty(A) (sizeof (A) > 1 ? sizeof (A) : 2)

/* Non-compliant - tries to match argument array type */
#define only_strings_nc(X) _Generic((X) \
 , char[SizeofNonEmpty (X)]: handle_sized_string (sizeof (X), (X)) \
 , char[1] : handle_null_terminator (1, (X)))

/* Compliant - matches pointer to argument array type */
#define only_strings(X) _Generic(&(X) \
 , char (*) [SizeofNonEmpty (X)]: handle_sized_string (sizeof (X), (X)) \
 , char (*) [1]: handle_null_terminator (1, (X)))

only_strings_nc ("hello"); /* Constraint violation */
only_strings ("world"); /* Matches as char (*) [6] */
```

## See also
Rule 23.5

---

Copyright The MISRA Consortium Limited © [Date - April 2023].
