# Rule 23.7 A generic selection that is expanded from a macro should evaluate its argument only once

## Category
Advisory

## Analysis
Decidable, Single Translation Unit

## Applies to
C11

## Amplification
If the controlling expression of a generic selection is expanded from a macro argument, it should also
be expanded elsewhere in the macro body so that it is evaluated exactly once in the entire expanded
macro body. The number of times that the expression is evaluated should not depend on which
association is selected, and should be consistent for all associations in the generic selection.

This rule does not depend on the presence of side effects in the operand expression.

## Rationale
The controlling expression of a generic selection is never evaluated. It is only used for its type, and
usually has to be repeated in order to be combined with the result value in some way.

If an expression is specified which syntactically contains a side effect, that effect will not be applied. If
the generic selection is (as is usually the case) the result of a macro expansion which uses one of the
macro's arguments to select the type, this non-evaluation is concealed from the invoking code.

Combining the input value with the result expression “outside” the generic selection (such as
choosing a function to call as the result, rather than the complete function call) is more likely to
guarantee consistent evaluation of the expanded operand.

## Exception
This rule is not violated if all result expressions for the generic selection are constant expressions,
and the macro never expands the argument outside of the controlling expression. This allows for the
implementation of type queries like `is_pointer_const`.

## Example
The following examples are consistent in their expansion of side effects for all associations. In the
first example, the argument used for the controlling expression are expanded exactly once outside of
the generic selection, so the generic selection does not prevent it from being evaluated. In the
second example, the argument expands once each into the result expression of each association. It
will be evaluated exactly once regardless of which generic association is selected.
```c
/* Compliant */
#define gfun1(X) ( _Generic((X) \
 , float32_t: fun1f \
 , float64_t: fun1 \
 , default : fun1l) (X) )

#define gfun2(X) _Generic((X) \
 , float32_t: fun2f (X) \
 , float64_t: fun2 (X) \
 , default : fun2l (X) )
```

The following example is non-compliant because whether or not the macro argument is evaluated
depends on whether the default association is selected or not. The default association does not
evaluate it because it has no use for the value, but this dangerously assumes that there were no side
effects in the expression.
```c
/* Non-compliant */
#define gfun3(X) _Generic((X) \
 , float32_t : fun3f (X) \
 , float64_t : fun3 (X) \
 , float128_t: fun3l (X) \
 , default : default_result )
```

The following example only expands a macro argument into the controlling expression, and all
possible result expressions are integer constant expressions. This implements a type query that can
be used for type checking.
```c
/* Compliant by exception */
#define is_pointer_const(P) _Generic(1 ? (P) : (void *)(P) \
 , void const *: 1 \
 , void const volatile *: 1 \
 , default : 0 )

_Static_assert (is_pointer_const (pi), "must not be an out-parameter");
```

## See also
Rule 23.2

---

Copyright The MISRA Consortium Limited © [Date - April 2023].
