# Rule 12.4 Evaluation of constant expressions should not lead to unsigned integer wrap-around

## Category
Advisory

## Analysis
Decidable, Single Translation Unit

## Applies to
C90, C99, C11

## Amplification
This rule applies to expressions that satisfy the *constraints* and semantics for a *constant expression*, whether or not they appear in a context that requires a *constant expression*.

If an expression is not evaluated, for example because it appears in the right operand of a logical AND operator whose left operand is always false, then this rule does not apply.

## Rationale
Unsigned integer expressions do not strictly overflow, but instead wrap-around. Although there may be good reasons to use modulo arithmetic at run-time, it is less likely for its use to be intentional at compile-time.

## Example
The expression associated with a *case* label is required to be a *constant expression*. If an unsigned wrap-around occurs during evaluation of a *case* expression, it is likely to be unintentional. On a machine with a 16-bit *int* type, any value of `BASE` greater than or equal to 65 024 would result in wrap-around in the following example:

```c
#define BASE 65024u

switch ( x )
{
  case BASE + 0u:
    f ( );
    break;
  case BASE + 1u:
    g ( );
    break;
  case BASE + 512u: /* Non-compliant - wraps to 0 */
    h ( );
    break;
}
```

The controlling expression of a *#if* or *#elif* preprocessor directive is required to be a *constant expression*:

```c
#if 1u + ( 0u - 10u ) /* Non-compliant as ( 0u - 10u ) wraps */
```

In this example, the expression `DELAY + WIDTH` has the value 70 000 but this will wrap-around to 4 464 on a machine with a 16-bit *int* type:

```c
#define DELAY 10000u
#define WIDTH 60000u

void fixed_pulse ( void )
{
  uint16_t off_time16 = DELAY + WIDTH; /* Non-compliant */
}
```

This rule does not apply to the expression `c + 1` in the following compliant example as it accesses an object and therefore does not satisfy the semantics for a *constant expression*:

```c
const uint16_t c = 0xffffu;

void f ( void )
{
  uint16_t y = c + 1u; /* Compliant */
}
```

In the following example, the sub-expression `( 0u - 1u )` leads to unsigned integer wrap-around. In the initialization of `x`, the sub-expression is not evaluated and the expression is therefore compliant. However, in the initialization of `y`, it may be evaluated and the expression is therefore non-compliant.

```c
bool_t b;

void g ( void )
{
  uint16_t x = ( 0u == 0u ) ? 0u : ( 0u - 1u ); /* Compliant     */
  uint16_t y = b ? 0u : ( 0u - 1u );            /* Non-compliant */
}
```

---

Copyright The MISRA Consortium Limited © [Date - March 2025].