<h2>Why is this an issue?</h2>
<p>Both <code>if-else</code> chains and <code>switch</code> statements are used for conditional branching, but they differ in their syntax and the way
they handle multiple conditions.</p>
<ul>
  <li> In an <code>if-else</code> chain, each condition is checked in order, and only the block associated with the first true condition is executed.
  If no condition is true, the code inside the <code>else</code> block (if present) will be executed. </li>
  <li> In a <code>switch</code> statement, the expression is evaluated once, and its value is compared against each case. If a matching case is found,
  the corresponding block of code is executed. The <code>break</code> statement is used to exit the <code>switch</code> block after a match. If no
  case matches the expression, the code inside the <code>default</code> block (if present) will be executed. </li>
</ul>
<p>Having the same condition in both <code>if-else</code> chains and <code>switch</code> cases can lead to unreachable code and a potential source of
bugs. It defeats the purpose of conditional branching and can make the code harder to read and maintain.</p>

```ts
if (event === 1) {
  openWindow();
} else if (event === 2) {
  closeWindow();
} else if (event === 1) {  // Noncompliant: Duplicated condition 'event === 1'
  moveWindowToTheBackground();
}

switch (event) {
  case 1:
    openWindow();
    break;
  case 2:
    closeWindow();
    break;
  case 1: // Noncompliant: Duplicated case '1'
    moveWindowToTheBackground();
    break;
}
```
<p>Carefully review your conditions and ensure that they are not duplicated across the <code>if-else</code> chain or <code>switch</code> statement.
Use distinct conditions and default blocks to cover all scenarios without redundant checks.</p>

```ts
if (event === 1) {
  openWindow();
} else if (event === 2) {
  closeWindow();
} else if (event === 3) {
  moveWindowToTheBackground();
}

switch (event) {
  case 1:
    openWindow();
    break;
  case 2:
    closeWindow();
    break;
  case 3:
    moveWindowToTheBackground();
    break;
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/if...else"><code>if...else</code></a>
  </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/switch"><code>switch</code></a> </li>
  <li> Wikipedia - <a href="https://en.wikipedia.org/wiki/Unreachable_code">Unreachable code</a> </li>
</ul>