<h2>Why is this an issue?</h2>
<p>React components should not be nested, as their state will be lost on each re-render of their parent component, possibly introducing bugs. This
will also impact performance as child components will be recreated unnecessarily.</p>
<p>If the goal is to have the state reset, use a <a
href="https://react.dev/learn/preserving-and-resetting-state#option-2-resetting-state-with-a-key"><code>key</code></a> instead of relying on a parent
state.</p>

```ts
function Component() {
  function NestedComponent() { // Noncompliant: NestedComponent should be moved outside Component
    return <div />;
  }

  return (
    <div>
      <NestedComponent />
    </div>
  );
}
```

```ts
function Component() {
  return (
    <div>
      <OtherComponent footer={ () => <div /> } /> { /* Noncompliant: Component is created inside prop */ }
    </div>
  );
}
```

```ts
class Component extends React.Component {
  render() {
    function UnstableNestedComponent() { // Noncompliant: NestedComponent should be moved outside Component
      return <div />;
    }

    return (
      <div>
        <UnstableNestedComponent />
      </div>
    );
  }
}
```
<p>You should refactor your code to define a component independently, passing props if needed.</p>

```ts
function OutsideComponent(props) {
  return <div />;
}

function Component() {
  return (
    <div>
      <OutsideComponent />
    </div>
  );
}
```

```ts
function Component() {
  return <OtherComponent footer={ <div /> } />;
}
```

```ts
class Component extends React.Component {
  render() {
    return (
      <div>
        <OtherComponent />
      </div>
    );
  }
}
```
<p>Component creation is allowed inside component props only if prop name starts with <code>render</code>. Make sure you are calling the prop in the
receiving component and not using it as an element.</p>

```ts
function OtherComponent(props) {
  return <div>{props.renderFooter()}</div>;
}

function Component() {
  return (
    <div>
      <OtherComponent renderFooter={() => <div />} />
    </div>
  );
}
```
<h2>Resources</h2>
<ul>
  <li> <a href="https://reactjs.org/docs/reconciliation.html#elements-of-different-types">React Documentation - Elements Of Different Types</a> </li>
  <li> <a href="https://react.dev/learn/preserving-and-resetting-state#option-2-resetting-state-with-a-key">React Documentation - Resetting state with
  a key</a> </li>
</ul>