/*
 * Decompiled with CFR 0.152.
 */
package org.sonarsource.sonarlint.core.serverapi.stream;

import com.google.common.util.concurrent.MoreExecutors;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import javax.annotation.Nullable;
import org.sonarsource.sonarlint.core.commons.log.SonarLintLogger;
import org.sonarsource.sonarlint.core.http.HttpClient;
import org.sonarsource.sonarlint.core.http.HttpConnectionListener;
import org.sonarsource.sonarlint.core.serverapi.ServerApiHelper;
import org.sonarsource.sonarlint.core.serverapi.stream.Event;
import org.sonarsource.sonarlint.core.serverapi.stream.EventBuffer;
import org.sonarsource.sonarlint.core.serverapi.stream.EventParser;

public class EventStream {
    private static final SonarLintLogger LOG = SonarLintLogger.get();
    private static final Integer UNAUTHORIZED = 401;
    private static final Integer FORBIDDEN = 403;
    private static final Integer NOT_FOUND = 404;
    private static final long HEART_BEAT_PERIOD = 60L;
    private final ServerApiHelper helper;
    private final ScheduledExecutorService executor;
    private final AtomicReference<HttpClient.AsyncRequest> currentRequest = new AtomicReference();
    private final AtomicReference<ScheduledFuture<?>> pendingFuture = new AtomicReference();
    private final Consumer<Event> eventConsumer;

    public EventStream(ServerApiHelper helper, Consumer<Event> eventConsumer) {
        this(helper, eventConsumer, Executors.newScheduledThreadPool(1));
    }

    EventStream(ServerApiHelper helper, Consumer<Event> eventConsumer, ScheduledExecutorService executor) {
        this.helper = helper;
        this.eventConsumer = eventConsumer;
        this.executor = executor;
    }

    public EventStream connect(String wsPath) {
        return this.connect(wsPath, new Attempt());
    }

    private EventStream connect(final String wsPath, final Attempt currentAttempt) {
        LOG.debug("Connecting to server event-stream at '" + wsPath + "'...");
        EventBuffer eventBuffer = new EventBuffer();
        this.currentRequest.set(this.helper.getEventStream(wsPath, new HttpConnectionListener(){

            @Override
            public void onConnected() {
                LOG.debug("Connected to server event-stream");
                EventStream.this.schedule(() -> EventStream.this.connect(wsPath), 180L);
            }

            @Override
            public void onError(@Nullable Integer responseCode) {
                EventStream.this.handleError(wsPath, currentAttempt, responseCode);
            }

            @Override
            public void onClosed() {
                EventStream.this.cancelPendingFutureIfAny();
                LOG.debug("Disconnected from server event-stream, reconnecting now");
                EventStream.this.connect(wsPath);
            }
        }, message -> {
            this.cancelPendingFutureIfAny();
            eventBuffer.append((String)message).drainCompleteEvents().forEach(stringEvent -> this.eventConsumer.accept(EventParser.parse(stringEvent)));
        }));
        return this;
    }

    private void handleError(String wsPath, Attempt currentAttempt, @Nullable Integer responseCode) {
        if (EventStream.shouldRetry(responseCode)) {
            if (!currentAttempt.isMax()) {
                long retryDelay = currentAttempt.delay;
                StringBuilder msgBuilder = new StringBuilder();
                msgBuilder.append("Cannot connect to server event-stream");
                if (responseCode != null) {
                    msgBuilder.append(" (").append(responseCode).append(")");
                }
                msgBuilder.append(", retrying in ").append(retryDelay).append("s");
                LOG.debug(msgBuilder.toString());
                this.schedule(() -> this.connect(wsPath, currentAttempt.next()), retryDelay);
            } else {
                LOG.debug("Cannot connect to server event-stream, stop retrying");
            }
        }
    }

    private static boolean shouldRetry(@Nullable Integer responseCode) {
        if (UNAUTHORIZED.equals(responseCode)) {
            LOG.debug("Cannot connect to server event-stream, unauthorized");
            return false;
        }
        if (FORBIDDEN.equals(responseCode)) {
            LOG.debug("Cannot connect to server event-stream, forbidden");
            return false;
        }
        if (NOT_FOUND.equals(responseCode)) {
            LOG.debug("Server events not supported by the server");
            return false;
        }
        return true;
    }

    private void schedule(Runnable task, long delayInSeconds) {
        if (!this.executor.isShutdown()) {
            this.pendingFuture.set(this.executor.schedule(task, delayInSeconds, TimeUnit.SECONDS));
        }
    }

    public void close() {
        this.cancelPendingFutureIfAny();
        HttpClient.AsyncRequest currentRequestOrNull = this.currentRequest.getAndSet(null);
        if (currentRequestOrNull != null) {
            currentRequestOrNull.cancel();
        }
        if (!MoreExecutors.shutdownAndAwaitTermination((ExecutorService)this.executor, (long)5L, (TimeUnit)TimeUnit.SECONDS)) {
            LOG.warn("Unable to stop event stream executor service in a timely manner");
        }
    }

    private void cancelPendingFutureIfAny() {
        ScheduledFuture pendingFutureOrNull = this.pendingFuture.getAndSet(null);
        if (pendingFutureOrNull != null) {
            pendingFutureOrNull.cancel(true);
        }
    }

    private static class Attempt {
        private static final int DEFAULT_DELAY_S = 60;
        private static final int BACK_OFF_MULTIPLIER = 2;
        private static final int MAX_ATTEMPTS = 10;
        private final long delay;
        private final int attemptNumber;

        public Attempt() {
            this(60L, 1);
        }

        public Attempt(long delay, int attemptNumber) {
            this.delay = delay;
            this.attemptNumber = attemptNumber;
        }

        public Attempt next() {
            return new Attempt(this.delay * 2L, this.attemptNumber + 1);
        }

        public boolean isMax() {
            return this.attemptNumber == 10;
        }
    }
}

