Boolean input parameters are often an indicator that a method does two things 
instead of one.

Splitting such methods may simplify their code and provide a more descriptive 
interface for callers.

# What Does This Check Look For?

This check reports findings for `IMPORTING` method parameters of type `abap_bool`.

# Why is This a Problem?

Boolean input parameters are an indicator that the method might do different things.
Moreover, if the parameter is the only one to the method, method calls may omit the
parameter name, so its meaning might be hard to understand from only looking at the 
method call.

Setter methods with a single boolean input parameter are acceptable, because none of
the above arguments apply to them.

# How Can I Resolve This?

In order to make it obvious what the boolean parameter stands for, and to have 
single responsibility in methods, split up methods that do different things depending
on the boolean value.

# Examples
## Non-Compliant

```ABAP
METHODS update
  IMPORTING
    do_save TYPE abap_bool.

" more code
    
update( abap_true ).            " The meaning of 'true' is not obvious.
```

## Compliant

```ABAP
METHODS set_is_deleted
  IMPORTING
    new_value TYPE abap_bool.   " This is ok, as it is a setter method.

METHODS update_without_saving.
METHODS update_and_save.

" more code

update_without_saving( ).       " The call's intent is much clearer now.
update_and_save( ).             " The call's intent is much clearer now.
```

# Where Can I Learn More?

- [Clean ABAP: Boolean Input Parameter(s)](https://github.com/SAP/styleguides/blob/main/clean-abap/CleanABAP.md#split-method-instead-of-boolean-input-parameter)
- [code pal for ABAP: Boolean Input Parameter](https://github.com/SAP/code-pal-for-abap/blob/master/docs/checks/boolean-input-parameter.md)
- [Wikipedia: Single-responsibility principle](https://en.wikipedia.org/wiki/Single-responsibility_principle)
