Using parallel streams can inadvertently increase code complexity and add the need for careful synchronization. Consider alternatives to parallel streams that do not raise these concerns.

# What Does This Check Look For?

This check identifies instances where parallel streams are utilized within the codebase.

# Why is This a Problem?

The utilization of parallel streams introduces additional complexity due to the required synchronization and potential threading issues. While they can offer performance benefits under the right circumstances, their misuse can lead to hard-to-diagnose bugs, decreased code comprehensibility, and potential performance degradation due to inappropriate use cases or environments. This goes against the best practices of keeping code simple, maintainable, and comprehensible.

# How Can I Resolve This?

Evaluate the necessity of using parallel streams, considering the context and requirements of your application:
- If the performance improvement with parallel streams is marginal or the task does not benefit from parallelism, replace parallel streams with sequential ones.
- Consider other concurrency mechanisms or data processing frameworks that provide more explicit control over parallelism and threading, if high performance for concurrent tasks is critical.
- Review and ensure proper synchronization when parallel streams are the only viable option, to mitigate potential threading issues.

This approach simplifies the codebase, reduces the risk of concurrency-related bugs, and improves maintainability.

# Examples

## Non-Compliant

```java
public static void someCalculation(List<Integer> dataList) {
	Map<Integer, Integer> resultMap = new HashMap<>();
	dataList.parallelStream().map(f -> {
		resultMap.put(f, 42); // not thread-safe
		return 0;
	}).collect(Collectors.toList());
}
```

## Compliant

```java
public static void someCalculation(List<Integer> dataList) {
	Map<Integer, Integer> resultMap = new HashMap<>();
	dataList.stream().map(f -> {
		resultMap.put(f, 42);
		return 0;
	}).collect(Collectors.toList());
}
```

# Where Can I Learn More?

- [Java API Specification: Collection.parallelStream](https://docs.oracle.com/en/java/javase/11/docs/api/java.base/java/util/Collection.html#parallelStream())