This check detects appearances of the `out` modifier for function parameters.

# Why is This a Problem?
The `out` modifier indicates that a function argument is passed by reference instead of value.
This way, the function can modify the value of the given parameter outside the function's scope using its pointer.

Handling pointers correctly requires experience with pointers and reference types causing the usage of `out` parameters to be error-prone.

# How Can I Resolve This?
Instead of changing values outside the function with `out` parameters, one should use dedicated `return` values.

# Examples
## Non-Compliant

C#:

```cs
void nonCompliantOneParam(out int returnValue) {
    returnValue += someInt;
}

void nonCompliantMultipleParam(out int returnValue, out int returnOtherValue) {
    returnValue += someInt;
    otherReturnValue += someInt;
}
```

PL/SQL:

```oraclesqlplus
function my_function (out_date out date) return boolean is
begin
    out_date := sysdate;
    return true;
end my_function;
```

## Compliant

C#:

```cs
int compliantOneParam(int value) {
    return value + someInt;
}

(int, int) compliantMultipleParam(int value, otherValue) {
    returnValue = value + someInt;
    returnOtherValue = otherValue + someInt;

    return (returnValue, returnOtherValue);
}

void callingFunction(int value, int otherValue) {
    value = compliantOneParam(value);
    (value, otherValue) = compliantMultipleParam(value, otherValue);
}
```

PL/SQL:

```oraclesqlplus
function my_function return date is
begin
  return sysdate;
end my_function;
```

# Where Can I Learn More?

## Guidelines
- [PL/SQL & SQL Coding Guidelines: G-7440: Never use OUT parameters to return values from a function.](https://trivadis.github.io/plsql-and-sql-coding-guidelines/v4.0/4-language-usage/7-stored-objects/4-functions/g-7440/)
- [Microsoft Dotnet Quality Rules: CA1021: Avoid out parameters](https://learn.microsoft.com/en-us/dotnet/fundamentals/code-analysis/quality-rules/ca1021)

## References

- [C# Reference: out parameter modifier](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/out-parameter-modifier)
