Static fields that are not thread-safe can cause unpredictable data issues in a multi-threaded environment.
Consider using thread-local storage or ensuring the field's thread safety manually.

# What Does This Check Look For?

This check identifies instances where non-thread-safe classes such as `Calendar`, `DateFormat`, `XPath`, or `SchemaFactory` are declared as `static` fields in Java.

# Why is This a Problem?

Non-thread-safe static fields, when accessed from multiple threads, may result in inconsistent state, data corruption, or unexpected exceptions.
Static fields are shared across all instances of a class, and the non-thread-safe nature of certain standard Java library classes can lead to serious concurrency issues if not handled carefully.
Multi-threaded applications rely on thread safety to ensure integrity and correctness of data processing.
Ignoring thread safety in such applications might not only lower performance but could also lead to critical data inaccuracies.

# How Can I Resolve This?

To resolve this issue, consider the following options based on your specific use case:
- Use `ThreadLocal` to create thread-confined instances of such objects, ensuring that each thread has its own instance, thus preventing concurrent access issues.
- If global access is necessary, synchronize access to these fields manually, using synchronization mechanisms provided by the Java language (e.g., `synchronized` methods or blocks, `ReentrantLocks`).
- Evaluate whether these static fields can be refactored into instance fields, reducing the risk of concurrency issues by design.

# Examples

## Non-Compliant

```java
public class DateFormatter {
	private static DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd"); // non-compliant: DateFormat is not thread-safe
	public static String formatDate(Date date) {
		return dateFormat.format(date);
	}
}
```

## Compliant

```java
public class ThreadSafeDateFormatter {
	private static final ThreadLocal<DateFormat> dateFormat =
			ThreadLocal.withInitial(() -> new SimpleDateFormat("yyyy-MM-dd")); // compliant: each thread gets its own DateFormat instance
	public static String formatDate(Date date) {
		return dateFormat.get().format(date);
	}
}
```

# Where Can I Learn More?

- [What Is Thread-Safety and How to Achieve It?](https://www.baeldung.com/java-thread-safety)
- Guideline: Oracle Java Documentation: [ThreadLocal](https://docs.oracle.com/javase/8/docs/api/java/lang/ThreadLocal.html)