Throwing generic exceptions hampers the ability to handle errors effectively and precisely.
Use specific exception types that accurately represent the error conditions.

# What Does This Check Look For?

This check identifies instances in Java code where generic exceptions are thrown, such as `Throwable`, `Error`, `RuntimeException`, and `Exception`.

# Why is This a Problem?

Throwing generic exceptions like `Throwable`, `Error`, `RuntimeException`, and `Exception` can make error handling more challenging and lead to less robust code:
1. **Lack of Specificity.** Generic exceptions fail to communicate the specific nature of the error, making it harder for developers to understand and address the issue.
2. **Error Handling Difficulties.** It encourages catch-all error handling, which can silently swallow unexpected errors or lead to a one-size-fits-all approach to error recovery.
3. **Decreased Code Clarity.** The use of generic exceptions makes it unclear what kinds of errors might occur, reducing code readability and maintainability.

# How Can I Resolve This?

Prefer throwing specific exceptions that accurately reflect the nature of the error condition. 
This approach aids in writing more targeted catch blocks that can handle specific error scenarios appropriately. 
By using specific exceptions, you also make your code easier to understand and maintain, as the exceptions thrown are self-documenting regarding the potential error states.
Besides specific exceptions from the package `java.lang` like `ArithmeticException` or `IllegalArgumentException` you can also define your own custom exceptions.

# Examples

## Non-Compliant

```java
public void readFile(String filePath) throws Exception {
    if (filePath == null) {
        throw new Exception("File path cannot be null.");
    }
    // additional logic
}
```

## Compliant

```java
public void readFile(String filePath) throws FileNotFoundException, IllegalArgumentException {
    if (filePath == null) {
        throw new IllegalArgumentException("File path cannot be null.");
    }
    // additional logic to open and read the file, potentially throwing FileNotFoundException
}
```

# Where Can I Learn More?

- [SEI CERT Oracle Coding Standard for Java: Exceptional Behavior](https://wiki.sei.cmu.edu/confluence/display/java/ERR07-J.+Do+not+throw+RuntimeException,+Exception,+or+Throwable)
- [Oracle Java Tutorials: Unchecked Exceptions - The Controversy](https://docs.oracle.com/javase/tutorial/essential/exceptions/runtime.html)
- [Tutorialspoint: Java.lang.Exceptions](https://www.tutorialspoint.com/java/lang/java_lang_exceptions.htm)
- [Baeldung: Create a Custom Exception in Java](https://www.baeldung.com/java-new-custom-exception)