This check detects  `WHERE ... LIKE` clauses of `SELECT`s whose search term contains a variable,
but which do not use the `ESCAPE` addition.

# Why is This a Problem?

`WHERE ... LIKE` queries allow the wildcard symbols `_` or `%` to match any character or substring, respectively.
This is convenient when the search term is statically defined, but can cause unintended results if the search term
is provided from outside, e.g. user input. In that case, these characters are probably meant to be taken literally
in the query. So, if e.g. `%` occurs in the search term, the `LIKE` does not only match the symbol `%` itself, but
any substring. This might be unintended, e.g. when the user wanted to search for values containing “42%”.

# How Can I Resolve This?

It is recommended to escape variable search strings first, especially when they are user input or if it is not
entirely clear where the input comes from. Then, in the `WHERE ... LIKE` clause, the `ESCAPE` addition should be
used.

In the following example, a search string pattern is created from an externally provided search term and a suffix
that contains the wildcard characters `%` and `_`. These should obviously be treated as wildcards. But the result
of `some->get_search_term()` should be taken literally. Since we cannot be sure that it doesn’t contain these
wildcard characters itself, it should be escaped. In the example, this is done by replacing all occurrences of
`_` or `%` in the search term with `#_` or `#%` before building the pattern. The `SELECT` statement then specifies
`#` as the `ESCAPE` character for the `LIKE` clause.

## Example (Before)

~~~
DATA(pattern) = |{ some->get_search_term() }%xy_|.
SELECT * FROM z_my_table WHERE my_field LIKE @pattern INTO TABLE @DATA(result).
~~~

## Example (After)

~~~
DATA(escaped_search_term) = replace(
val   = some->get_search_term()
regex = `(_|%)`
with  = `#$1`
occ   = 0 ).

DATA(pattern) = |{ escaped_search_term }%xy_|.
SELECT * FROM z_my_table WHERE my_field LIKE @pattern ESCAPE '#' INTO TABLE @DATA(result).
~~~

# References

[SAP ABAP Keyword Documentation: sql_cond - LIKE](https://help.sap.com/doc/abapdocu_750_index_htm/7.50/en-US/abenwhere_logexp_like.htm)

[SAP ABAP Keyword Documentation: wildcard characters](https://help.sap.com/doc/abapdocu_750_index_htm/7.50/en-US/abenwildcard_character_glosry.htm)