# Rule 13.2 The value of an expression and its persistent side effects shall be the same under all permitted evaluation orders and shall be independent from thread interleaving
## Category
Required
## Analysis
Undecidable, System
## Applies to
C90, C99, C11
## Amplification
Between any two adjacent sequence points:
1. No object shall be modified more than once;
2. All parts of the expression are considered when determining whether an object is read or modified, irrespective of any known values.
3. No object shall be both modified and read unless any such read of the object's value contributes towards computing the value to be stored into the object;
4. There shall be no more than one modification access with *volatile*-qualified or atomic type;
5. There shall be no more than one read access with *volatile*-qualified type;
6. There shall be no more than one read access to an object with atomic type.
Notes:
1. An object might be accessed indirectly, by means of a pointer or a called function, as well as being accessed directly by the expression.
2. This Amplification is intentionally stricter than the headline of the rule. As a result, expressions such as `x = x = 0;` are not permitted by this rule even though the value and the *persistent side effects*, provided that `x` is not *volatile*, are independent of the order of evaluation or *side effects*.
Sequence points are summarized in Annex C of the C Standard. The sequence points in C90 are a subset of those in later editions.
## Rationale
The C Standard gives considerable flexibility to compilers when evaluating expressions. Most operators can have their operands evaluated in any order. The main exceptions are:
- Logical AND `&&` in which the second operand is evaluated only if the first operand evaluates to non-zero;
- Logical OR `||` in which the second operand is evaluated only if the first operand evaluates to zero;
- The conditional operator `?:` in which the first operand is always evaluated and then either the second or third operand is evaluated;
- The `,` operator in which the first operand is evaluated and then the second operand is evaluated.
Note: The presence of parentheses may alter the order in which operators are applied. However, this does not affect the order of evaluation of the lowest-level operands, which may be evaluated in any order.
The atomic types provide assurance that a single read or write access to an atomic object is not subject to interruption or potential interference from other threads. However, that does not prevent two distinct atomic accesses to the same variable by a thread being pre-empted by another thread modifying that variable. On non-atomic variables such interference can only be caused by data races and constitute undefined behaviour. By definition, although there are no data races on atomic variables, such interference is still undesirable.
Many of the common instances of the unpredictable behaviour, associated with expression evaluation, can be avoided by following the advice given by this rule, by Rule 13.3, and by Rule 13.4. However, in order to simplify this rule, it does restrict some forms which are well-defined.
## Example
```c
/* When the COPY_ELEMENT macro is invoked in this non-compliant example, i is read twice and modified twice. It is unspecified whether the order of operations on i is:
 * Read, modify, read, modify, or
 * Read, read, modify, modify.
 */
#define COPY_ELEMENT( index ) ( a[( index )] = b[( index )] )
COPY_ELEMENT ( i++ );

/* In this non-compliant example the order in which v1 and v2 are read is unspecified. */
extern volatile uint16_t v1, v2;
uint16_t t;
t = v1 + v2;

/* In this compliant example PORT is read and modified. */
extern volatile uint8_t PORT;
PORT = PORT & 0x80u;

/* The order of evaluation of function arguments is unspecified as is the order in which side effects occur as shown in this non-compliant example. */
uint16_t i = 0;
/* 
 * Unspecified whether this call is equivalent to:
 * f ( 0, 0 )
 * or f ( 0, 1 )
 */
f ( i++, i );

/* The relative order of evaluation of a function designator and function arguments is unspecified. In this non-compliant example, if the call to g modifies p then it is unspecified whether the function designator p->f uses the value of p prior to the call of g or after it. */
p->f ( g ( &p ) );

/* In the following example, Thread T2 might interrupt Thread T1 while the expression a - a is evaluated. Then the first load instruction for a loads the value 10, but the second load operation loads the value 7. The compliant solution avoids the problem by storing the value of a in a local variable. */
_Atomic int32_t a;

int32_t t1( void* ignore ) /* Thread T1 entry */
{
 int32_t v1, v2;
 int32_t acopy; 
 a = 10;
 acopy = a; /* acopy may be either 10 or 7 */
 v1 = a - a; /* Non-compliant - v1 may be 0 or 3 */
 v2 = acopy - acopy; /* Compliant - v2 is always 0 */
 return v1 + v2;
}

int32_t t2( void* ignore ) /* Thread T2 entry */
{
 a = 7;
 
 return a;
}
```
## See also
Dir 4.9, Rule 13.1, Rule 13.3, Rule 13.4

---

Copyright The MISRA Consortium Limited © [Date - April 2023].
