# Rule 19.3 A union member shall not be read unless it has been previously set

## Category
Required

## Analysis
Undecidable, System

## Applies to
C90, C99, C11

## Amplification
When a union member is accessed, the value of that member shall have previously been set by initialization or assignment, and no assignment to a different member shall have occurred in between this assignment and the value access.

This rule applies recursively to the members of any objects of structure types that are top-level members of the union itself.

## Rationale
If a union member is written and then a different union member is read back, the behaviour is implementation-defined:
*   if the member read back is wider that the member written, the value is unspecified;
*   in C90, the behaviour is undefined;
*   otherwise, the value obtained is implementation-defined and may not be a valid object representation.

Intentionally accessing a different member from the member most recently written to in this way is called *punning*.

If the member written to is smaller than the whole union, the remaining bytes of the union take on unspecified values. Therefore, accessing a larger member than the member most recently written to is equivalent to accessing a partially-uninitialized object.

Aliasing reads through character types are guaranteed by the C Standard to result in a valid individual-byte access; there is no equivalent guarantee for aliasing writes.

## Exception
This rule does not apply if the union member read from is a character array member.

## Example
In this non-compliant example, a floating point number is stored into a *union* and an integer is read back, implying an attempt to inspect the representation of the *float*. The returned integer value is not guaranteed to have any meaningful or useful value:
```c
uint32_t floatrep( float flt )
{
  union
  {
    float flt;
    uint32_t bits;
  } tmp;

  tmp.flt = flt;
  return tmp.bits; /* Non-compliant - read of different member */
}
```
In this non-compliant example, a 16-bit integer is stored into a *union* that previously stored a 32-bit integer, and then a 32-bit integer is read back. Although it was written to, the trailing bytes of `tmp.x` became unspecified when `tmp.lo` was written. This therefore also violates Rule 9.1:
```c
uint32_t zerobits( uint32_t x )
{
  union
  {
    uint32_t x;
    uint16_t lo;
  } tmp;

  tmp.x = x;
  tmp.lo = 0;
  return tmp.x; /* Non-compliant - x was not written most recently */
}
```
In this compliant example, a *float* is stored into a *union* and an individual byte is read back from the representation of the *float*. Although the value of the byte is implementation-defined, the access is guaranteed to be valid:
```c
uint8_t floatbytes( float flt )
{
  union
  {
    float flt;
    uint8_t bytes[ sizeof(float) ];
  } tmp;

  tmp.flt = flt;
  return tmp.bytes[1]; /* Compliant by exception - individual byte access */
}
```

## See also
Rule 9.1, Rule 19.2

---

Copyright The MISRA Consortium Limited © [Date - March 2025].