# Rule 21.25 All memory synchronization operations shall be executed in sequentially consistent order

## Category
Required

## Analysis
Decidable, Single Translation Unit

## Applies to
C11

## Amplification
The Standard provides an enumerated type *memory_order* to specify the behaviour of memory synchronization operations. Only the memory order *memory_order_seq_cst* shall be used.

The following library functions implicitly use memory ordering *memory_order_seq_cst*:
`atomic_store`, `atomic_load`, `atomic_flag_test_and_set`, `atomic_flag_clear`,
`atomic_exchange`, `atomic_compare_exchange_strong`, `atomic_compare_exchange_weak`,
`atomic_fetch_add`, `atomic_fetch_sub`, `atomic_fetch_or`, `atomic_fetch_xor`,
`atomic_fetch_and`

For each of these functions, there exists an alternate version with the function name ending in *_explicit()*, which takes an explicit *memory_order* parameter. The functions ending in *_explicit()* shall only be called with the enumeration *memory_order_seq_cst* as the *memory_order* parameter.

Also the following functions shall only be called with the enumeration *memory_order_seq_cst* as the *memory_order* parameter:
`atomic_thread_fence`, `atomic_signal_fence`

## Rationale
The Standard defines *memory_order_seq_cst* as the default memory order for objects with atomic types. This ordering is fully defined in the C Standard and enables sequential consistency. The behaviour of other memory orders is non-portable, as it depends on hardware architecture and compiler.

For *memory_order_relaxed*, no operation orders memory. Usage of *memory_order_relaxed* can cause unintuitive behaviour and is error-prone.

Many of those library functions listed above impose restrictions on the memory order allowed, e.g. it is undefined behaviour if the *atomic_store* generic function is called with a *memory_order_acquire*, *memory_order_consume*, or *memory_order_acq_rel* order argument. In case of non-compliant usage, compilers may show warnings but still generate code.

## Example
```c
typedef struct s {
  uint8_t a;
  uint8_t b;
} s_t;
_Atomic s_t astr;

void main( void ) 
{
  s_t lstr = {7, 42};
  
  atomic_init( &astr, lstr );
  
  lstr = atomic_load( &astr ); /* Compliant */
  lstr = atomic_load_explicit( &astr, memory_order_relaxed ); /* Non-compliant */
  lstr.b = 43;
  atomic_store_explicit( &astr, lstr, memory_order_release ); /* Non-compliant */
}
```

## See also
Dir 4.13

---

Copyright The MISRA Consortium Limited © [Date - March 2025].