# Rule 0.0.1 A function shall not contain *unreachable* statements

## Category
Required

## Analysis
Decidable, Single Translation Unit

## Amplification
A statement is *unreachable* if the block containing it is not *reachable* from the entry block of the Control
Flow Graph (CFG) for the function.

For the purpose of this rule:
*   Both operands of a *reachable* logical AND (`&&`) or logical OR (`||`) operator are considered
    *reachable*; and
*   All three operands of a *reachable* conditional operator (`?:`) are considered *reachable*; and
*   The blocks linked by the edges from a *condition* of a *selection-statement* or an *iteration-statement*
    are all considered *reachable*, except when the *condition* is a *constant expression*, in which case
    only the blocks linked by edges selected by the *condition* are considered *reachable*; and
*   A call to a function declared `[[noreturn]]` has no CFG out edge; and
*   If a `try` *compound-statement* of a *(function-)try-block* does not contain a *reachable*, *potentially-
    throwing* statement, then all *catch-handlers* are *unreachable*, otherwise all *catch-handlers* are
    considered *reachable* subject to the restriction that a *catch-handler* that appears after a more
    generic *handler* of the same *try-block* is not *reachable*.

The rule does not apply to *statements* in the discarded branch of a `constexpr if` statement.

## Rationale
*Unreachable* code often indicates a defect in the program, as, assuming that the program does not
exhibit any *undefined behaviour*, *unreachable* code cannot be executed and cannot have any effect on
the program’s outputs.

In order to avoid crosstalk with Rule 0.0.2, the handling of logical and conditional operators in the
conceptual CFG used by this rule differs from that in a traditional CFG.

## Example
```cpp
bool f0();

int32_t f1( int32_t c, int32_t & res )
{
  if ( false && f0() ) { } // Compliant - statement is considered to be reachable

  return res;

  res = c; // Non-compliant - not reachable from entry block

  bool result; // Non-compliant - not reachable from entry block
}

void f2( int32_t i )
{
  while ( true ) // Constant condition - single edge into body of loop
  {
    if ( i != 0 )
    {
      break; // Adds edge to statements following the loop body
    }
  }

  ++i; // Compliant - reachable via 'break'

  while ( true ) // Constant condition - single edge into body of loop
  {
    f();
  }

  ++i; // Non-compliant - not reachable from entry block
}

void f3( int32_t i )
{
  goto LABEL;
  ++i; // Non-compliant - no edge to this block

LABEL:
  ++i; // Compliant
}

class BaseException {};
class DerivedException: public BaseException {};

void f4()
{
  try { /* ... */ }
  catch ( BaseException & b ) { }
  catch ( DerivedException & d ) { } // Non-compliant - will be caught above
}

void f5() noexcept;

void f6()
{
  try { f5(); }
  catch ( int32_t ) { } // Non-compliant - f5 is not potentially-throwing
  catch ( ... ) { } // Non-compliant - f5 is not potentially-throwing
}

void f7( int32_t i )
{
  try
  {
    throw i;
    ++i; // Non-compliant - no edge to this block
  }
  catch ( int32_t ) { } // Compliant - all catch-handlers are reachable
  catch ( int16_t ) { } // Compliant - all catch-handlers are reachable

  ++i; // Compliant
}

void f8();

int32_t f9( int32_t i )
{
  try
  {
    f8(); // Potentially-throwing
    return i * 2; // Compliant
  }
  catch( int32_t ) { } // Compliant - all catch-handlers are reachable

  return 0; // Compliant - even if f8 throws a type
} // other than int32_t

[[noreturn]] void f10() noexcept;

int32_t f11()
{
  f10(); // Does not return
  return 0; // Non-compliant
}
```

---

Copyright The MISRA Consortium Limited © [Date - October 2023].
