# Rule 0.0.2 Controlling expressions should not be invariant

## Category
Advisory

## Analysis
Undecidable, System

## Amplification
This rule applies to:
*   Controlling expressions of `if`, `while`, `for`, `do ... while` and `switch` statements; and
*   The first operand of the conditional operator (`?:`); and
*   The left hand operand of the logical AND (`&&`) and logical OR (`||`) operators.

It does not apply to controlling expressions of `constexpr if` statements.

A function’s compliance with this rule is determined independently of the context in which the function
is called. For example, a Boolean parameter is treated as if it may have a value of `true` or `false`,
even if all the calls expressed in the current program use a value of `true`.

## Rationale
If a controlling expression has an invariant value, it is possible that there is a programming error. Any
code in an *infeasible path* may be removed by the compiler, which might have the effect of removing
code that has been introduced for defensive purposes.

This rule does not apply to `constexpr if`, as this is intended to be evaluated at compile time and
requires a constant expression.

## Exception
1.  A `while` statement with a constant expression evaluating to `true` is permitted as this is
    commonly used in real time systems.
2.  Macros are permitted to expand to a *do-while* statement of the form `do { } while ( false )`,
    allowing a macro expansion to be used as a statement that includes a local scope.

## Example
```cpp
s8a = ( u16a < 0u ) ? 0 : 1; // Non-compliant - u16a always >= 0

if ( u16a <= 0xffffu ) { } // Non-compliant - always true
if ( 2 > 3 ) { } // Non-compliant - always false
if ( ( s8a < 10 ) && ( s8a > 20 ) ) { } // Non-compliant - always false
if ( ( s8a < 10 ) || ( s8a > 5 ) ) { } // Non-compliant - always true
if ( ( s8a < 10 ) &&
     ( s8a > 20 ) || // Non-compliant - left operand of ||
     ( s8b == 5 ) ) { } // always false

const uint8_t N = 4u;

if ( N == 4u ) // Non-compliant - compiler is permitted
{ // to assume that N always has value 4
}

extern const volatile uint8_t M;

if ( M == 4u ) // Compliant - compiler assumes M may
{ // change, even though the program
} // cannot modify its value

while ( s8a > 10 )
{
  if ( s8a > 5 ) { } // Non-compliant - s8a always > 5

  --s8a;
}

for ( s8a = 0; s8a < 130; ++s8a ) { } // Non-compliant - always true

while ( true ) { /* Do something */ } // Compliant by exception #1

do { } while ( false ); // Compliant by exception #2
                        // - if expanded from a macro

uint16_t n; // Assume 10 <= n <= 100
uint16_t sum;

sum = 0;

for ( uint16_t i = ( n - 6u ); i < n; ++i )
{
  sum += i;
}

if ( ( sum % 2u ) == 0u )
{
  // Non-compliant - the sum of six, consecutive, non-negative integers is always
  // an odd number, so the controlling expression will always be false.
}

template< typename T >
void foo()
{
  if constexpr ( std::is_integral< T >() ) // Rule does not apply
  {
    // Handle integral case
  }
  else
  {
    // Handle other case
  }
}

template void foo< int >();
template void foo< float >();
```

## See also
Rule 0.0.1

---

Copyright The MISRA Consortium Limited © [Date - October 2023].
