# Rule 0.2.1 Variables with *limited visibility* should be *used* at least once

## Category
Advisory

## Analysis
Decidable, Single Translation Unit

## Amplification
A variable has *limited visibility* if it is not a function parameter, and it has internal linkage or no linkage.

A variable is *used* when:
1.  It is part of an *id-expression*; or
2.  The variable is of class type and has a *user-provided* constructor or a *user-provided* destructor.

## Rationale
Variables that are declared and never *used* within a project do not contribute to program output; they
constitute noise and may indicate that the wrong variable name has been *used* or that one or more
statements are missing.

Note: this rule allows the introduction of variables for the sole purpose of providing scoped resource
allocation and release. For example:
```cpp
{
  std::lock_guard< std::mutex > lock { mutex }; // Compliant - has user-provided
                                                // constructor
  // ...
} // User-provided destructor implicitly called here
```

## Exception
This rule does not apply to:
1.  Variables that have at least one *declaration* with the `[[maybe_unused]]` attribute.
2.  Constant variables at namespace scope that are declared within a *header file*.

## Example
```cpp
class C { }; // No user-provided constructor or destructor

namespace
{
  C c; // Non-compliant - unused
}

void maybeUnused( int32_t a )
{
  [[maybe_unused]]
  bool b = a > 0; // Compliant (by exception #1 if NDEBUG is defined)

  assert( b ); // Does not use b if NDEBUG is defined

  usefn( a );
}

const int16_t x = 19; // Compliant - x is read in initializedButNotUsed
const int16_t y = 21; // Non-compliant - would be compliant by exception #2
                      // if declared in a header file

void initializedButNotUsed()
{
  int16_t local_1 = 42; // Non-compliant - local_1 is never read
  int16_t local_2; // Compliant

  local_2 = x; // Use of local_2 for the purposes of this rule
}

void userProvidedCtor()
{
  std::ifstream fs { "cfg.ini" }; // Compliant - user-provided constructor
}
```

## See also
Rule 0.2.2

---

Copyright The MISRA Consortium Limited © [Date - October 2023].
