<h2>Why is this an issue?</h2>
<p>Using the equality <code>==</code> and inequality <code>!=</code> operators to compare two objects generally works. The operators can be
overloaded, and therefore the comparison can resolve to the appropriate method. However, when the operators are used on interface instances, then
<code>==</code> resolves to reference equality, which may result in unexpected behavior if implementing classes override <code>Equals</code>.
Similarly, when a class overrides <code>Equals</code>, but instances are compared with non-overloaded <code>==</code>, there is a high chance that
value comparison was meant instead of the reference one.</p>
<h3>Noncompliant code example</h3>
<pre>
public interface IMyInterface
{
}

public class MyClass : IMyInterface
{
    public override bool Equals(object obj)
    {
        //...
    }
}

public class Program
{
    public static void Method(IMyInterface instance1, IMyInterface instance2)
    {
        if (instance1 == instance2) // Noncompliant, will do reference equality check, but was that intended? MyClass overrides Equals.
        {
            Console.WriteLine("Equal");
        }
    }
}
</pre>
<h3>Compliant solution</h3>
<pre>
public interface IMyInterface
{
}

public class MyClass : IMyInterface
{
    public override bool Equals(object obj)
    {
        //...
    }
}

public class Program
{
    public static void Method(IMyInterface instance1, IMyInterface instance2)
    {
        if (object.Equals(instance1, instance2)) // object.Equals checks for null and then calls the instance based Equals, so MyClass.Equals
        {
            Console.WriteLine("Equal");
        }
    }
}
</pre>
<h3>Exceptions</h3>
<p>The rule does not report on comparisons of <code>System.Type</code> instances and on comparisons inside <code>Equals</code> overrides.</p>
<p>It also does not raise an issue when one of the operands is <code>null</code> nor when one of the operand is cast to <code>object</code> (because
in this case we want to ensure reference equality even if some <code>==</code> overload is present).</p>
<h2>Resources</h2>
<ul>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/595">CWE-595 - Comparison of Object References Instead of Object Contents</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/597">CWE-597 - Use of Wrong Operator in String Comparison</a> </li>
</ul>

