<h2>Why is this an issue?</h2>
<p>When you do not use the return value of a method with no side effects, it indicates that something is wrong. Either this method is unnecessary, or
the source code does not behave as expected and could lead to code defects. For example, there are methods, such as <a
href="https://learn.microsoft.com/en-us/dotnet/api/system.datetime.addyears">DateTime.AddYears</a>, that don’t change the value of the input object,
but instead, they return a new object whose value is the result of this operation, and as a result that you will have unexpected effects if you do not
use the return value.</p>
<p>This rule raises an issue when the results of the following methods are ignored:</p>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/concepts/linq/">LINQ</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.contracts.pureattribute"><code>Pure</code> methods</a> </li>
  <li> Any method on <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/built-in-types">build-in types</a>
  </li>
  <li> Any method on <a href="https://learn.microsoft.com/en-us/archive/msdn-magazine/2017/march/net-framework-immutable-collections">Immutable
  collections</a> </li>
</ul>
<p>Special cases:</p>
<ul>
  <li> Although <a href="https://learn.microsoft.com/en-us/dotnet/api/system.string.intern"><code>string.Intern</code></a> has a side effect, ignoring
  its return value is still suspicious as it is the only reference ensured to point to the intern pool. </li>
  <li> LINQ methods can have side effects if they are misused. For example: </li>
</ul>
<pre>
data.All(x =&gt;
{
    x.Property = "foo";
    return true;
});
</pre>
<p>Such code should be rewritten as a loop because <a
href="https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.all"><code>Enumerable.All&lt;TSource&gt;</code></a> method should be used to
determine if all elements satisfy a condition and not to change their state.</p>
<h3>Exceptions</h3>
<p>This rule doesn’t report issues on invocations with <a
href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/out-parameter-modifier"><code>out</code></a> or <a
href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/ref"><code>ref</code></a> arguments.</p>
<h2>How to fix it</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
data.Where(x =&gt; x &gt; 5).Select(x =&gt; x * x); // Noncompliant
"this string".Equals("other string"); // Noncompliant

data.All(x =&gt;  // Noncompliant
{
    x.Property = "foo";
    return true;
});
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
var res = data.Where(x =&gt; x &gt; 5).Select(x =&gt; x * x);
var isEqual = "this string".Equals("other string");

foreach (var x in data)
{
    x.Property = "foo";
}
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.contracts.pureattribute"><code>PureAttribute</code>
  Class</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/out-parameter-modifier"><code>out</code>
  parameter modifier</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/ref"><code>ref</code> keyword</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/api/system.string.intern"><code>String.Intern(String)</code> Method</a>
  </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/concepts/linq/">LINQ</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/built-in-types">build-in types</a>
  </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/archive/msdn-magazine/2017/march/net-framework-immutable-collections">Immutable
  collections</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> <a href="https://www.daniellittle.dev/dont-ignore-your-functions">Don’t ignore your functions</a> </li>
</ul>

