<h2>Why is this an issue?</h2>
<p>A shared resource refers to a resource or data that can be accessed or modified by multiple <a
href="https://en.wikipedia.org/wiki/Thread_(computing)">threads</a> or concurrent parts of a program. It could be any piece of data, object, file,
database connection, or system resource that needs to be accessed or manipulated by multiple parts of a program at the same time.</p>
<p>Shared resources should not be used for <a href="https://en.wikipedia.org/wiki/Lock_(computer_science)">locking</a> because it increases the chance
of <a href="https://en.wikipedia.org/wiki/Deadlock">deadlocks</a>. Any other thread could acquire (or attempt to acquire) the same lock while doing
some operation, without knowing that the resource is meant to be used for locking purposes.</p>
<p>For example, a <code>string</code> should never be used for locking. When a <code>string</code> is <a
href="https://en.wikipedia.org/wiki/Interning_(computer_science)">interned</a> by the runtime, it can be shared by multiple threads, breaking the
locking mechanism.</p>
<p>Instead, a dedicated private <code>object</code> instance should be used for each shared resource. This minimizes access to the lock instance,
avoiding deadlocks and lock contention.</p>
<p>The following objects are considered as shared resources:</p>
<ul>
  <li> a reference to <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/this">this</a>: if the instance is publicly
  accessibly, the lock might be shared </li>
  <li> a <a href="https://learn.microsoft.com/en-us/dotnet/api/system.type">Type</a> object: if the type class is publicly accessibly, the lock might
  be shared </li>
  <li> a <a href="https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/strings/">string</a> literal or instance: if any other part of the
  program uses the same string, the lock is shared because of interning </li>
</ul>
<h2>How to fix it</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
void MyLockingMethod()
{
    lock (this) // Noncompliant
    {
        // ...
    }
}
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
private readonly object lockObj = new object();
void MyLockingMethod()
{
    lock (lockObj)
    {
        // ...
    }
}
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://en.wikipedia.org/wiki/Thread_(computing)">Thread</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Lock_(computer_science)">Locking</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Deadlock">Deadlock</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Interning_(computer_science)">Interning</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.string.intern#remarks">String interning by the runtime</a> </li>
  <li> <a href="https://docs.microsoft.com/en-us/dotnet/standard/threading/managed-threading-best-practices">Managed Threading Best Practices</a>
  </li>
</ul>

