<p>Gratuitous boolean expressions are conditions that do not change the evaluation of a program. This issue can indicate logical errors and affect the
correctness of an application, as well as its maintainability.</p>
<h2>Why is this an issue?</h2>
<p>Control flow constructs like <code>if</code>-statements allow the programmer to direct the flow of a program depending on a boolean expression.
However, if the condition is always true or always false, only one of the branches will ever be executed. In that case, the control flow construct and
the condition no longer serve a purpose; they become <em>gratuitous</em>.</p>
<h3>What is the potential impact?</h3>
<p>The presence of gratuitous conditions can indicate a logical error. For example, the programmer <em>intended</em> to have the program branch into
different paths but made a mistake when formulating the branching condition. In this case, this issue might result in a bug and thus affect the
reliability of the application. For instance, it might lead to the computation of incorrect results.</p>
<p>Additionally, gratuitous conditions and control flow constructs introduce unnecessary complexity. The source code becomes harder to understand, and
thus, the application becomes more difficult to maintain.</p>
<p>This rule looks for operands of a boolean expression never changing the result of the expression. It also applies to the <a
href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/null-coalescing-operator">null coalescing operator</a> when one of
the operands always evaluates to <code>null</code>.</p>
<pre>
string d = null;
var v1 = d ?? "value";      // Noncompliant
</pre>
<h3>Exceptions</h3>
<p>This rule will not raise an issue in either of these cases:</p>
<ul>
  <li> When the condition is a single <code>const bool</code> <pre>
const bool debug = false;
//...
if (debug)                  // Compliant
{
  // Print something
}
</pre> </li>
  <li> When the condition is the literal <code>true</code> or <code>false</code>. </li>
</ul>
<p>In these cases, it is obvious the code is as intended.</p>
<h2>How to fix it</h2>
<p>Gratuitous boolean expressions are suspicious and should be carefully removed from the code.</p>
<p>First, the boolean expression in question should be closely inspected for logical errors. If a mistake was made, it can be corrected so the
condition is no longer gratuitous.</p>
<p>If it becomes apparent that the condition is actually unnecessary, it can be removed. The associated control flow construct (e.g., the
<code>if</code>-statement containing the condition) will be adapted or even removed, leaving only the necessary branches.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre>
public void Sample(bool b, bool c)
{
    var a = true;
    if (a)                  // Noncompliant: "a" is always "true"
    {
        DoSomething();
    }

    if (b &amp;&amp; a)             // Noncompliant: "a" is always "true"
    {
        DoSomething();
    }

    if (c || !a)            // Noncompliant: "!a" is always "false"
    {
        DoSomething();
    }

    string d = null;
    var v1 = d ?? "value";  // Noncompliant: "d" is always null and v1 is always "value".
    var v2 = s ?? d;        // Noncompliant: "d" is always null and v2 is always equal to s.
}
</pre>
<h4>Compliant solution</h4>
<p>The unnecessary operand is updated:</p>
<pre>
public void Sample(bool b, bool c, string s)
{
    var a = IsAllowed();
    if (a)                  // Compliant
    {
        DoSomething();
    }

    if (b &amp;&amp; a)             // Compliant
    {
        DoSomething();
    }

    if (c || !a)            // Compliant
    {
        DoSomething();
    }

    string d = GetStringData();
    var v1 = d ?? "value";  // Compliant
    var v2 = s ?? d;        // Compliant
}
</pre>
<p>The unnecessary operand is removed:</p>
<pre>
public void Sample(bool b, bool c, string s)
{
    DoSomething();

    if (b)                  // Compliant
    {
        DoSomething();
    }

    if (c)                  // Compliant
    {
        DoSomething();
    }

    var v1 = "value";       // Compliant
    var v2 = s;             // Compliant
}
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/571">CWE-571 - Expression is Always True</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/570">CWE-570 - Expression is Always False</a> </li>
  <li> <a
  href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/boolean-logical-operators#conditional-logical-and-operator-">Conditional logical AND operator &amp;&amp;</a> </li>
  <li> <a
  href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/boolean-logical-operators#conditional-logical-or-operator-">Conditional logical OR operator ||</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/null-coalescing-operator">Null-coalescing operators ??
  and ??=</a> </li>
</ul>

