<p>You should only set a property of a temporal type (like <code>DateTime</code> or <code>DateTimeOffset</code>) as the primary key of a table if the
values are guaranteed to be unique.</p>
<h2>Why is this an issue?</h2>
<p>Using temporal types as the primary key of a table is risky. When these types are used as primary keys, it usually means that each new key is
created with the use of <code>.Now</code> or <code>.UtcNow</code> properties from <code>DateTime</code> and <code>DateTimeOffset</code> classes. In
those cases, duplicate keys exceptions may occur in many ways:</p>
<ul>
  <li> when entries are added consecutively by a machine with low-enough system clock resolution; </li>
  <li> when two different threads are inserting entries in close enough sequence for both to have the same time; </li>
  <li> when changes such as daylight saving time (DST) transitions occur, where values can be repeated the following hour (only for
  <code>DateTime</code> type); </li>
</ul>
<p>The rule raises an issue if:</p>
<ul>
  <li> Entity Framework, or Entity Framework Core dependencies are found; </li>
  <li> a class contains a property either named <code>Id</code>, <code>&lt;type name&gt;Id</code> or decorated by the <code>[Key]</code> or
  <code>[PrimaryKey]</code> attribute. </li>
  <li> the key property is of one of the following types:
    <ul>
      <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.datetime">System.DateTime</a> </li>
      <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.datetimeoffset">System.DateTimeOffset</a> </li>
      <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.timespan">System.TimeSpan</a> </li>
      <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.dateonly">System.DateOnly</a> </li>
      <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.timeonly">System.TimeOnly</a> </li>
    </ul>  </li>
</ul>
<h2>How to fix it</h2>
<p>Either use a GUID or the auto generated ID as a primary key.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
internal class Account
{
    public DateTime Id { get; set; }

    public string Name { get; set; }
    public string Surname { get; set; }
}
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
internal class Account
{
    public Guid Id { get; set; }

    public string Name { get; set; }
    public string Surname { get; set; }
}
</pre>
<p>or</p>
<h4>Noncompliant code example</h4>
<pre data-diff-id="2" data-diff-type="noncompliant">
internal class Person
{
    [Key]
    public DateTime PersonIdentifier { get; set; }

    public string Name { get; set; }
    public string Surname { get; set; }
}
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="2" data-diff-type="compliant">
internal class Person
{
    [Key]
    public Guid PersonIdentifier { get; set; }

    public string Name { get; set; }
    public string Surname { get; set; }
}
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/ef/core/modeling/keys?tabs=data-annotations">Entity Framework keys and data annotation</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.guid">GUID</a> </li>
</ul>

