<p>Since .NET 6 you don’t have to use the <code>TimeZoneConverter</code> library to manually do the conversion between IANA and Windows timezones. The
.NET 6.0 introduced new Time Zone enhancements, one being the <code>TimeZoneInfo.FindSystemTimeZoneById(string timezone)</code> method now accepts as
input both IANA and Windows time zone IDs on any operating system with installed time zone data. <code>TimeZoneInfo.FindSystemTimeZoneById</code> will
automatically convert its input from IANA to Windows and vice versa if the requested time zone is not found on the system.</p>
<h2>Why is this an issue?</h2>
<p>The method <code>TimeZoneInfo.FindSystemTimeZoneById(string timezone)</code> can get both IANA and Windows timezones as input and automatically
convert one to the other if the requested time zone is not found on the system. Because one does not need to handle the conversion, the code will be
less complex and easier to maintain.</p>
<h2>How to fix it</h2>
<p>There’s no need to translate manually between time zones; it is enough to call <code>TimeZoneInfo.FindSystemTimeZoneById(string timezone)</code>,
where the timezone can be IANA or Windows format. Depending on the OS, the equivalent time zone will be returned (Windows Time Zones for Windows and
IANA timezones for Linux, macOS).</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
// Assuming we are in Windows OS and we need to get the Tokyo Time Zone.
var ianaTimeZone = "Asia/Tokyo";
var windowsTimeZone = TZConvert.IanaToWindows(ianaTimeZone);
TimeZoneInfo tokyoWindowsTimeZone = TimeZoneInfo.FindSystemTimeZoneById(windowsTimeZone);
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
// Assuming we are in Windows OS and we need to get the Tokyo Time Zone.
var ianaTimeZone = "Asia/Tokyo";
TimeZoneInfo tokyoWindowsTimeZone = TimeZoneInfo.FindSystemTimeZoneById(ianaTimeZone);
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.timezoneinfo.findsystemtimezonebyid">TimeZoneInfo.FindSystemTimeZoneById
  documentation</a> </li>
  <li> <a href="https://devblogs.microsoft.com/dotnet/date-time-and-time-zone-enhancements-in-net-6/">Date, Time, and Time Zone Enhancements in .NET
  6</a> </li>
  <li> <a href="https://github.com/mattjohnsonpint/TimeZoneConverter">TimeZoneConverter</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> <a href="https://codeblog.jonskeet.uk/2022/02/05/whats-up-with-timezoneinfo-on-net-6-part-1/">What’s up with TimeZoneInfo on .NET 6?</a> </li>
</ul>
<h3>Standards</h3>
<ul>
  <li> <a href="https://www.iana.org/time-zones">IANA Time Zone Database</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/windows-hardware/manufacture/desktop/default-time-zones?view=windows-11">Windows Time Zones</a>
  </li>
</ul>

