<h2>Why is this an issue?</h2>
<p>A key facet of the <code>equals</code> contract is that if <code>a.equals(b)</code> then <code>b.equals(a)</code>, i.e. that the relationship is
symmetric.</p>
<p>Using <code>instanceof</code> breaks the contract when there are subclasses, because while the child is an <code>instanceof</code> the parent, the
parent is not an <code>instanceof</code> the child. For instance, assume that <code>Raspberry extends Fruit</code> and adds some fields (requiring a
new implementation of <code>equals</code>):</p>

```java
Fruit fruit = new Fruit();
Raspberry raspberry = new Raspberry();

if (raspberry instanceof Fruit) { ... } // true
if (fruit instanceof Raspberry) { ... } // false
```
<p>If similar <code>instanceof</code> checks were used in the classes' <code>equals</code> methods, the symmetry principle would be broken:</p>

```java
raspberry.equals(fruit); // false
fruit.equals(raspberry); //true
```
<p>Additionally, non <code>final</code> classes shouldn’t use a hardcoded class name in the <code>equals</code> method because doing so breaks the
method for subclasses. Instead, make the comparison dynamic.</p>
<p>Further, comparing to an unrelated class type breaks the contract for that unrelated type, because while
<code>thisClass.equals(unrelatedClass)</code> can return true, <code>unrelatedClass.equals(thisClass)</code> will not.</p>
<h3>Noncompliant code example</h3>

```java
public class Fruit extends Food {
  private Season ripe;

  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (Fruit.class == obj.getClass()) { // Noncompliant; broken for child classes
      return ripe.equals(((Fruit)obj).getRipe());
    }
    if (obj instanceof Fruit ) {  // Noncompliant; broken for child classes
      return ripe.equals(((Fruit)obj).getRipe());
    }
    else if (obj instanceof Season) { // Noncompliant; symmetry broken for Season class
      // ...
    }
    //...
```
<h3>Compliant solution</h3>

```java
public class Fruit extends Food {
  private Season ripe;

  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (this.getClass() == obj.getClass()) {
      return ripe.equals(((Fruit)obj).getRipe());
    }
    return false;
}
```
<h2>Resources</h2>
<ul>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/AzZGBQ">CERT, MET08-J.</a> - Preserve the equality contract when overriding the equals() method
  </li>
</ul>