<h2>Why is this an issue?</h2>
<p>Java 21 has introduced enhancements to switch statements and expressions, allowing them to operate on any type, not just specific ones, as in
previous versions. Furthermore, case labels have been upgraded to support patterns, providing an alternative to the previous restriction of only
accepting constants.</p>

```java
// As of Java 21
String patternMatchSwitch(Object obj) {
    return switch (obj) {
        case String s  -> String.format("String %s", s);
        case Integer i -> String.format("int %d", i);
        default        -> obj.toString();
    };
}
```
<p>This allows to use the <code>when</code> keyword to specify a condition for a case label, also called a guarded case label.</p>

```java
String guardedCaseSwitch(Object obj) {
    return switch (obj) {
        case String s when s.length() > 0 -> String.format("String %s", s);
        case Integer i when i > 0 -> String.format("int %d", i);
        default        -> obj.toString();
    };
}
```
<p>This syntax is more readable and less error-prone than using an if statement inside the case block and should be preferred.</p>
<p>This rule reports an issue when a single <code>if</code> statement is used inside a case block.</p>
<h2>How to fix it</h2>
<p>Replace the <code>if</code> statement with a guarded case label.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
void testObject(Object response) {
    switch (response) {
        case String s -> {
            if(s.length() > 80) { // Noncompliant; use the "when" keyword
                System.out.println("This string is too long");
            }
        }
        case Integer i -> {
            if(i > 80) { // Noncompliant; use the "when" keyword
                System.out.println("This integer is too big");
            }
        }
        default -> System.out.println("Unknown type");
    }
}
```
<h4>Compliant solution</h4>

```java
void testObject(Object response) {
    switch (response) {
        case String s when s.length() > 80 -> {
            System.out.println("This string is too long");
        }
        case Integer i when i > 80 -> {
            System.out.println("This integer is too big");
        }
        default -> System.out.println("Unknown type");
    }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Java Documentation - <a href="https://openjdk.org/jeps/441">Pattern Matching for switch</a> </li>
</ul>