<h2>Why is this an issue?</h2>
<p>The <code>finally</code> block is a part of a <code>try…​catch…​finally</code> statement, which allows you to handle errors and perform cleanup
operations regardless of whether an exception is thrown or not. The <code>finally</code> block is executed regardless of whether an exception occurs
or not, and it is placed after the <code>try</code> and <code>catch</code> blocks.</p>
<p>Having a jump statement, such as <code>return</code>, <code>break</code>, <code>continue</code>, or <code>throw</code>, inside a
<code>finally</code> block can lead to unexpected and undesirable behavior, making your code difficult to understand and maintain. While it’s not
inherently forbidden to use jump statements in <code>finally</code> blocks, it is generally discouraged for the following reasons:</p>
<ul>
  <li> The primary purpose of the <code>finally</code> block is to ensure cleanup operations and code that must run regardless of the outcome, such as
  releasing resources or closing connections. If you use a <code>return</code> statement inside the finally block, it will override any previous
  <code>return</code> statements in the <code>try</code> or <code>catch</code> blocks. This can lead to unexpected values being returned from a
  function. </li>
  <li> Jump statements like <code>break</code>, <code>continue</code>, or even another <code>throw</code> inside the <code>finally</code> block can
  alter the normal control flow of the program. This can make it difficult to reason about the behavior of the code and may introduce subtle bugs that
  are hard to detect. </li>
  <li> If a <code>return</code> or <code>throw</code> statement inside the <code>finally</code> block causes a new exception or alters the return
  value, it can hide or suppress the original exception or return value from the <code>try</code> or <code>catch</code> blocks. This can make it
  challenging to identify the actual cause of an error. </li>
  <li> Code that uses jump statements in <code>finally</code> blocks can be hard to read and understand, especially for other developers who might not
  be familiar with the unusual control flow. Such code can lead to maintenance issues and make it harder to debug and maintain the application in the
  long run. </li>
</ul>
<p>This rule reports on all usages of jump statements from a <code>finally</code> block. Even if it’s guaranteed that no unhandled exception can
happen in <code>try</code> or <code>catch</code> blocks, it’s not recommended to use any jump statements inside the <code>finally</code> block to have
the logic there limited to the "cleanup".</p>

```js
async function foo() {
    let result, connection;
    try {
        connection = await connect();
        result = connection.send(1);
    } catch(err) {
        console.error(err.message);
    } finally {
        if (connection) {
            connection.close();
        }
        return result; // Noncompliant: Jump statement 'return' in the 'finally' block
    }
}
```
<p>While there might be rare cases where using jump statements in a <code>finally</code> block is necessary, it’s generally recommended to avoid it
whenever possible. Instead, use the <code>finally</code> block only for cleanup operations and critical tasks that should always be executed,
regardless of exceptions or return values.</p>

```js
async function foo() {
    let result, connection;
    try {
        connection = await connect();
        result = connection.send(1);
    } catch(err) {
        console.error(err.message);
    } finally {
        if (connection) {
            connection.close();
        }
    }
    return result;
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/try...catch#the_finally_block">The
  <code>finally</code> block</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/try...catch"><code>try...catch</code></a>
  </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/return"><code>return</code></a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/throw"><code>throw</code></a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/break"><code>break</code></a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/continue"><code>continue</code></a> </li>
</ul>