<h2>Why is this an issue?</h2>
<p>Cloud platforms such as AWS, Azure, or GCP support virtual firewalls that can be used to restrict access to services by controlling inbound and
outbound traffic.<br> Any firewall rule allowing traffic from all IP addresses to standard network ports on which administration services
traditionally listen, such as 22 for SSH, can expose these services to exploits and unauthorized access.</p>
<h3>What is the potential impact?</h3>
<p>Like any other service, administration services can contain vulnerabilities. Administration services run with elevated privileges and thus a
vulnerability could have a high impact on the system.</p>
<p>Additionally, credentials might be leaked through phishing or similar techniques. Attackers who are able to reach the services could use the
credentials to log in to the system.</p>
<h2>How to fix it</h2>
<p>It is recommended to restrict access to remote administration services to only trusted IP addresses. In practice, trusted IP addresses are those
held by system administrators or those of <a href="https://aws.amazon.com/quickstart/architecture/linux-bastion/?nc1=h_ls">bastion-like</a>
servers.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<p>For <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_ec2.Instance.html">aws-cdk-lib.aws_ec2.Instance</a> and other constructs
that support a <code>connections</code> attribute:</p>

```js
import {aws_ec2 as ec2} from 'aws-cdk-lib'

const instance = new ec2.Instance(this, "default-own-security-group",{
    instanceType: nanoT2,
    machineImage: ec2.MachineImage.latestAmazonLinux(),
    vpc: vpc,
    instanceName: "test-instance"
})

instance.connections.allowFrom(
    ec2.Peer.anyIpv4(), // Noncompliant
    ec2.Port.tcp(22),
    /*description*/ "Allows SSH from all IPv4"
)
```
<p>For <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_ec2.SecurityGroup.html">aws-cdk-lib.aws_ec2.SecurityGroup</a></p>

```js
import {aws_ec2 as ec2} from 'aws-cdk-lib'

const securityGroup = new ec2.SecurityGroup(this, "custom-security-group", {
    vpc: vpc
})

securityGroup.addIngressRule(
    ec2.Peer.anyIpv4(), // Noncompliant
    ec2.Port.tcpRange(1, 1024)
)
```
<p>For <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_ec2.CfnSecurityGroup.html">aws-cdk-lib.aws_ec2.CfnSecurityGroup</a></p>

```js
import {aws_ec2 as ec2} from 'aws-cdk-lib'

new ec2.CfnSecurityGroup(
    this,
    "cfn-based-security-group", {
        groupDescription: "cfn based security group",
        groupName: "cfn-based-security-group",
        vpcId: vpc.vpcId,
        securityGroupIngress: [
            {
                ipProtocol: "6",
                cidrIp: "0.0.0.0/0", // Noncompliant
                fromPort: 22,
                toPort: 22
            }
        ]
    }
)
```
<p>For <a
href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_ec2.CfnSecurityGroupIngress.html">aws-cdk-lib.aws_ec2.CfnSecurityGroupIngress</a></p>

```js
import {aws_ec2 as ec2} from 'aws-cdk-lib'

new ec2.CfnSecurityGroupIngress( // Noncompliant
    this,
    "ingress-all-ip-tcp-ssh", {
        ipProtocol: "tcp",
        cidrIp: "0.0.0.0/0",
        fromPort: 22,
        toPort: 22,
        groupId: securityGroup.attrGroupId
})
```
<h4>Compliant solution</h4>
<p>For <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_ec2.Instance.html">aws-cdk-lib.aws_ec2.Instance</a> and other constructs
that support a <code>connections</code> attribute:</p>

```js
import {aws_ec2 as ec2} from 'aws-cdk-lib'

const instance = new ec2.Instance(this, "default-own-security-group",{
    instanceType: nanoT2,
    machineImage: ec2.MachineImage.latestAmazonLinux(),
    vpc: vpc,
    instanceName: "test-instance"
})

instance.connections.allowFrom(
    ec2.Peer.ipv4("192.0.2.0/24"),
    ec2.Port.tcp(22),
    /*description*/ "Allows SSH from a trusted range"
)
```
<p>For <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_ec2.SecurityGroup.html">aws-cdk-lib.aws_ec2.SecurityGroup</a></p>

```js
import {aws_ec2 as ec2} from 'aws-cdk-lib'

const securityGroup3 = new ec2.SecurityGroup(this, "custom-security-group", {
    vpc: vpc
})

securityGroup3.addIngressRule(
    ec2.Peer.anyIpv4(),
    ec2.Port.tcpRange(1024, 1048)
)
```
<p>For <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_ec2.CfnSecurityGroup.html">aws-cdk-lib.aws_ec2.CfnSecurityGroup</a></p>

```js
import {aws_ec2 as ec2} from 'aws-cdk-lib'

new ec2.CfnSecurityGroup(
    this,
    "cfn-based-security-group", {
        groupDescription: "cfn based security group",
        groupName: "cfn-based-security-group",
        vpcId: vpc.vpcId,
        securityGroupIngress: [
            {
                ipProtocol: "6",
                cidrIp: "192.0.2.0/24",
                fromPort: 22,
                toPort: 22
            }
        ]
    }
)
```
<p>For <a
href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_ec2.CfnSecurityGroupIngress.html">aws-cdk-lib.aws_ec2.CfnSecurityGroupIngress</a></p>

```js
new ec2.CfnSecurityGroupIngress(
    this,
    "ingress-all-ipv4-tcp-http", {
        ipProtocol: "6",
        cidrIp: "0.0.0.0/0",
        fromPort: 80,
        toPort: 80,
        groupId: securityGroup.attrGroupId
    }
)
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html">AWS Documentation</a> - Security groups for your VPC </li>
  <li> <a href="https://docs.microsoft.com/en-us/azure/virtual-network/network-security-groups-overview">Azure Documentation</a> - Network security
  groups </li>
  <li> <a href="https://cloud.google.com/vpc/docs/using-firewalls">GCP Documentation</a> - Firewalls </li>
</ul>
<h3>Standards</h3>
<ul>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/284">CWE-284 - Improper Access Control</a> </li>
</ul>