<h2>Why is this an issue?</h2>
<p>Assigning a value to an import variable will cause a runtime error and will raise a compilation error in TypeScript.</p>
<h3>Named imports</h3>
<p>When using named imports, the imported identifier is a <em>live binding</em> exported by another module. Live bindings can be updated or reassigned
by the exporting module, and the imported value would also change. The importing module cannot reassign it.</p>

```js
import { exportedObject } from 'module.js';
exportedObject = 'hello world!';   // Noncompliant: TypeError: Assignment to constant variable.
```
<p>This rule will <strong>not</strong> raise an issue when a module mutates the imported object. Be aware that all other modules importing the same
value will observe the mutated value.</p>

```js
import { exportedObject } from 'module.js';
exportedObject.newAttribute = 'hello world!'; // exportedObject now contains newAttribute and can be seen from all other modules importing it
```
<h3>Namespace and dynamic imports</h3>
<p>This rule will raise an issue when modifying members of a <a
href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/import#module_namespace_object"><em>module namespace
object</em></a>. A module namespace object is a <a
href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Object/seal">sealed object</a> that describes all exports from
a module.</p>
<p>This can be done using</p>
<ul>
  <li> a namespace import </li>
</ul>

```js
import * as module from 'module.js';
module.newObject = module.exportedObject; // Noncompliant: TypeError: Cannot add property readPath, object is not extensible
```
<ul>
  <li> the fulfillment value of a dynamic import. </li>
</ul>

```js
import('module.js').then(module => {
  module.newObject = module.exportedObject; // Noncompliant: TypeError: Cannot add property readPath, object is not extensible
})
```
<h3>Default imports</h3>
<p>Default imports are live bindings to the <code>default</code> export. As with the other forms of <code>import</code> declarations, the importing
module cannot reassign it.</p>

```js
import module from 'module.js';
module = 'hello world!';   // Noncompliant: TypeError: Assignment to constant variable.
```
<p>However, the object which <code>default</code> refers to is not a <em>live binding</em> and may still be mutated by importing modules.</p>

```js
import moduleDefault from 'module.js';
moduleDefault.newAttribute = 'hello world!'; // module.default now contains newAttribute and can be seen from all other modules importing it
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/import">import</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/import#module_namespace_object">Module
  namespace object</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Object/isSealed#description">Sealed
  Objects</a> </li>
</ul>