<h2>Why is this an issue?</h2>
<p>The optional chaining operator <code>?.</code> allows to access a deeply nested property, returning <code>undefined</code> if the property or any
intermediate object is <code>undefined</code>.</p>
<p>This usually means that the expression is expected to evaluate as <code>undefined</code> in some cases. Therefore, using the optional chaining
operator in a context where returning <code>undefined</code> throws an error can lead to runtime exceptions.</p>

```js
(event?.callback)(); // Noncompliant: when 'event' does not have 'callback' property TypeError is thrown
const { code } = event?.error; // Noncompliant: when 'event' does not have 'error' property TypeError is thrown
func(...event?.values); // Noncompliant: when 'event' does not have 'values' property TypeError is thrown
```
<p>Since optional chaining represents multiple execution branches, having an error thrown in such a context can be hard to debug.</p>
<h2>How to fix it</h2>
<p>In order to prevent runtime errors, you should provide fallbacks for when the optional chaining operator short-circuits to
<code>undefined</code>.</p>
<ul>
  <li> Nullish coalescing operator <code>??</code> </li>
</ul>

```js
(event?.callback ?? defaultCallback)();
```
<ul>
  <li> Logical OR operator <code>||</code> </li>
</ul>

```js
(event?.callback || defaultCallback)();
```
<ul>
  <li> Ternary operator <code>condition ? exprIfTrue : exprIfFalse</code> </li>
</ul>

```js
(event?.callback ? event?.callback : defaultCallback)();
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Optional_chaining">Optional chaining
  (?.)</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Nullish_coalescing">Nullish coalescing
  operator (??)</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Logical_OR">Logical OR (||)</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Conditional_operator">Conditional (ternary)
  operator</a> </li>
</ul>